#include "police/jani/parser/parser.hpp"

#include "police/base_types.hpp"
#include "police/compute_graph_factory.hpp"
#include "police/expressions/const_expression_folder.hpp"
#include "police/expressions/constant_fetcher.hpp"
#include "police/expressions/constants.hpp"
#include "police/expressions/expression_evaluator.hpp"
#include "police/expressions/expression_normalizer.hpp"
#include "police/expressions/expression_transformer.hpp"
#include "police/expressions/expression_visitor.hpp"
#include "police/expressions/expressions.hpp"
#include "police/expressions/filter_property.hpp"
#include "police/expressions/identifier_reference.hpp"
#include "police/expressions/properties.hpp"
#include "police/expressions/property.hpp"
#include "police/expressions/state_predicate_property.hpp"
#include "police/expressions/state_property.hpp"
#include "police/expressions/until_property.hpp"
#include "police/jani/model.hpp"
#include "police/jani/parser/automaton.hpp"
#include "police/jani/parser/composition.hpp"
#include "police/jani/parser/constant_declaration.hpp"
#include "police/jani/parser/constant_definitions.hpp"
#include "police/jani/parser/jani2nn.hpp"
#include "police/jani/parser/model.hpp"
#include "police/jani/parser/types.hpp"
#include "police/jani/parser/variable.hpp"
#include "police/layer_bounds.hpp"
#include "police/macros.hpp"
#include "police/storage/unordered_map.hpp"
#include "police/storage/unordered_set.hpp"
#include "police/storage/vector.hpp"
#include "police/utils/dependency_queue.hpp"
#include "police/utils/unordered_inserter.hpp"
#include "police/verification_property.hpp"

#if POLICE_VERITAS
#include <veritas/addtree.hpp>
#include <veritas/json_io.hpp>
#endif

#include <algorithm>
#include <cassert>
#include <filesystem>
#include <fstream>
#include <functional>
#include <iterator>
#include <limits>
#include <memory>
#include <ranges>
#include <string_view>
#include <sys/types.h>
#include <type_traits>
#include <variant>

namespace police::jani {

namespace {

using namespace police::jani::parser;

void check_no_continuous_types(const parser::Model& model)
{
    const auto check_type = [](const Type& t) {
        std::visit(
            [](auto&& val) -> void {
                using T = std::decay_t<decltype(val)>;
                if constexpr (std::is_same_v<T, ContinuousType>) {
                    POLICE_EXIT_INVALID_INPUT(
                        "continuous type is currently not supported");
                } else if constexpr (std::is_same_v<T, ClockType>) {
                    POLICE_EXIT_INVALID_INPUT(
                        "clock type is currently not supported");
                }
            },
            t);
    };
    const auto check_internal = [&](auto&& vec) {
        std::ranges::for_each(
            vec | std::views::transform(
                      [](const auto& constant) { return constant.type; }),
            check_type);
    };
    check_internal(model.constants);
    check_internal(model.variables);
    std::for_each(
        model.automata.begin(),
        model.automata.end(),
        [&](const auto& automaton) { check_internal(automaton.variables); });
}

void check_no_transient_variables(const parser::Model& model)
{
    auto has_transient = [](auto&& vars) {
        return std::any_of(vars.begin(), vars.end(), [](auto&& var) {
            return var.transient;
        });
    };
    if (has_transient(model.variables) ||
        std::any_of(
            model.automata.begin(),
            model.automata.end(),
            [&](auto&& automaton) {
                return has_transient(automaton.variables);
            })) {
        POLICE_EXIT_INVALID_INPUT(
            "transient variables are currently not supported");
    }
}

void check_not_input_enabled(const Composition& system)
{
    if (std::find_if(
            system.elements.begin(),
            system.elements.end(),
            [](auto&& elem) { return elem.input_enable.size() > 0; }) !=
        system.elements.end()) {
        POLICE_EXIT_INVALID_INPUT(
            "input enabled automata are currently not supported");
    }
}

void check_single_initial_location(const vector<parser::Automaton>& automata)
{
    if (std::any_of(
            automata.begin(),
            automata.end(),
            [](const parser::Automaton& automaton) {
                return automaton.initial_locations.size() != 1u;
            })) {
        POLICE_EXIT_INVALID_INPUT(
            "automata with multiple initial locations are "
            "currently not supported");
    }
}

void check_system_composition_not_empty(const Composition& composition)
{
    if (composition.elements.empty()) {
        POLICE_EXIT_INVALID_INPUT(
            "system composition must consist of at least one automaton");
    }
}

void check_sync_sizes(const Composition& composition)
{
    if (!composition.syncs.has_value()) return;
    const police::size_t expected_size = composition.elements.size();
    for (const auto& sync : composition.syncs.value()) {
        if (sync.synchronise.size() != expected_size) {
            POLICE_EXIT_INVALID_INPUT(
                "synchronization vector doesn't match system elements ("
                << sync.synchronise.size() << " vs " << expected_size << ")");
        }
    }
}

void make_syntactic_prechecks(const parser::Model& model)
{
    check_not_input_enabled(model.system);
    check_system_composition_not_empty(model.system);
    check_sync_sizes(model.system);
    check_no_continuous_types(model);
    check_no_transient_variables(model);
    check_single_initial_location(model.automata);
}

template <typename T, typename Project>
unordered_map<identifier_name_t, police::size_t>
get_index_map(const vector<T>& objs, Project&& project, std::string_view hint)
{
    unordered_map<identifier_name_t, police::size_t> indices;
    for (police::size_t i = 0; i < objs.size(); ++i) {
        if (indices.count(project(objs[i]))) {
            POLICE_EXIT_INVALID_INPUT(
                "multiply defined " << hint << " with name "
                                    << project(objs[i]));
        }
        indices[project(objs[i])] = i;
    }
    return indices;
}

auto get_constant_indices(const vector<ConstantDeclaration>& declarations)
{
    return get_index_map(
        declarations,
        [](const auto& autom) { return autom.identifier; },
        "constants");
}

class ConstantReferenceCollector final : public expressions::ExpressionVisitor {
public:
    explicit ConstantReferenceCollector(
        const unordered_map<identifier_name_t, police::size_t>* constant_id)
        : constant_id(constant_id)
    {
    }

    void visit(const expressions::IdentifierReference& expr) override
    {
        auto id = constant_id->find(expr.identifier);
        if (id == constant_id->end()) {
            POLICE_EXIT_INVALID_INPUT(
                "no constant with name " << expr.identifier << " defined");
        }
        constants.insert(id->second);
    }

    const unordered_map<identifier_name_t, police::size_t>* constant_id;
    unordered_set<police::size_t> constants;
};

DependencyQueue get_constant_dependencies(
    const vector<ConstantDeclaration>& declarations,
    const ConstantDefinitions& predefs,
    const unordered_map<identifier_name_t, police::size_t>& indices)
{
    vector<ConstantReferenceCollector> dependencies(
        declarations.size(),
        ConstantReferenceCollector(&indices));
    for (police::size_t i = 0; i < declarations.size(); ++i) {
        if (declarations[i].value) {
            if (predefs.contains(declarations[i].identifier)) {
                POLICE_EXIT_INVALID_INPUT(
                    "value given for constant "
                    << declarations[i].identifier
                    << " but is not a model parameter");
            }
            declarations[i].value->accept(dependencies[i]);
        } else if (!predefs.contains(declarations[i].identifier)) {
            POLICE_EXIT_INVALID_INPUT(
                "no value given for model parameter "
                << declarations[i].identifier);
        }
    }
    return DependencyQueue(
        std::ranges::views::transform(dependencies, [](const auto& dependency) {
            return dependency.constants;
        }));
}

vector<Value> get_constant_value_vector(
    const unordered_map<identifier_name_t, police::size_t>& indices,
    const ConstantDefinitions& predefs)
{
    vector<Value> values(indices.size(), Value(false));
    for (const auto& [name, value] : predefs) {
        const auto idx = indices.find(name);
        if (idx == indices.end()) {
            POLICE_EXIT_INVALID_INPUT(
                "parameter with name " << name
                                       << " does not reference a constant");
        }
        values[idx->second] = value;
    }
    return values;
}

struct ConstantLookup {
    ConstantLookup(
        const unordered_map<identifier_name_t, police::size_t>* indicies,
        const vector<Value>* values)
        : indices(indicies)
        , values(values)
    {
    }

    std::optional<Value> operator()(const identifier_name_t& ref) const
    {
        const auto index = indices->find(ref);
        if (index != indices->end()) {
            return values->at(index->second);
        }
        return std::nullopt;
    }

    const unordered_map<identifier_name_t, police::size_t>* indices;
    const vector<Value>* values;
};

vector<Value> evaluate_constants(
    const vector<ConstantDeclaration>& declarations,
    const unordered_map<identifier_name_t, police::size_t>& indices,
    const ConstantDefinitions& predefs)
{
    DependencyQueue queue =
        get_constant_dependencies(declarations, predefs, indices);
    auto values = get_constant_value_vector(indices, predefs);
    while (queue.can_pop()) {
        const police::size_t idx = queue.pop();
        assert(
            predefs.contains(declarations[idx].identifier) ||
            declarations[idx].value.has_value());
        if (declarations[idx].value.has_value()) {
            expressions::ExpressionEvaluator<ConstantLookup> evaluator(
                ConstantLookup(&indices, &values));
            declarations[idx].value.value().accept(evaluator);
            assert(evaluator.value.has_value());
            values[idx] = evaluator.value.value();
        }
    }
    if (!queue.empty()) {
        POLICE_EXIT_INVALID_INPUT(
            "could not evaluate all constant "
            "expressions: there is a cyclic dependency "
            "between some constants");
    }
    return values;
}

class ConstantRefInstantiator final
    : public expressions::ExpressionTransformer {
public:
    ConstantRefInstantiator(
        const unordered_map<identifier_name_t, police::size_t>& indices,
        const vector<Value>& values)
        : indices(&indices)
        , values(&values)
    {
    }

    void visit(Expression& ptr, expressions::IdentifierReference& ref) override
    {
        auto idx = indices->find(ref.identifier);
        if (idx != indices->end()) {
            ptr = expressions::MakeConstant()(values->at(idx->second));
        }
    }

    const unordered_map<identifier_name_t, police::size_t>* indices;
    const vector<Value>* values;
};

void process_constant_expressions(
    parser::Model& model,
    const ConstantDefinitions& constant_predefs)
{
    {
        const auto indices = get_constant_indices(model.constants);
        const auto values =
            evaluate_constants(model.constants, indices, constant_predefs);
        model.constants.clear();
        ConstantRefInstantiator instantiator(indices, values);
        transform_all(model, instantiator);
    }
    {
        expressions::ConstExpressionFolder simplifier;
        transform_all(model, simplifier);
    }
}

constexpr police::size_t DISCARD = std::numeric_limits<police::size_t>::max();

auto get_automata_indices(const vector<parser::Automaton>& automata)
{
    return get_index_map(
        automata,
        [](const auto& autom) { return autom.name; },
        "automata");
}

vector<police::size_t> get_automata_permutation_map(
    const vector<CompositionElement>& elements,
    const unordered_map<identifier_name_t, police::size_t>& indices)
{
    vector<police::size_t> permutation(elements.size(), DISCARD);
    for (police::size_t i = 0; i < elements.size(); ++i) {
        auto idx = indices.find(elements[i].automaton);
        if (idx == indices.end()) {
            POLICE_EXIT_INVALID_INPUT(
                "there is not automaton with name " << elements[i].automaton);
        }
        permutation[i] = idx->second;
    }
    return permutation;
}

void remove_unreferenced_automata(parser::Model& model)
{
    const auto indices = get_automata_indices(model.automata);
    const auto permutation =
        get_automata_permutation_map(model.system.elements, indices);
    vector<parser::Automaton> automata;
    automata.reserve(model.system.elements.size());
    for (const auto pos : permutation) {
        automata.push_back(model.automata[pos]);
    }
    automata.swap(model.automata);
}

unordered_set<identifier_name_t> get_actions(const vector<parser::Edge>& edges)
{
    unordered_set<identifier_name_t> actions;
    auto view =
        edges | std::ranges::views::filter([](auto&& edge) {
            return edge.action.has_value();
        }) |
        std::ranges::views::transform([](auto&& edge) -> identifier_name_t {
            return edge.action.value();
        });
    std::copy(view.begin(), view.end(), unordered_inserter(actions));
    return actions;
}

unordered_set<identifier_name_t>
get_relevant_actions(const Composition& system, police::size_t automaton)
{
    unordered_set<identifier_name_t> actions;
    for (const auto& sync : system.syncs.value()) {
        assert(automaton < sync.synchronise.size());
        const auto& label = sync.synchronise[automaton];
        if (label.has_value()) {
            actions.insert(label.value());
        }
    }
    return actions;
}

void check_action_validity(
    std::string_view automaton_name,
    const unordered_set<identifier_name_t>& relevant,
    const unordered_set<identifier_name_t>& all)
{
    auto it =
        std::find_if(relevant.begin(), relevant.end(), [&](auto&& action) {
            return !all.count(action);
        });
    if (it != relevant.end()) {
        POLICE_EXIT_INVALID_INPUT(
            "automaton " << automaton_name << " has not action named " << *it);
    }
}

void remove_unreferenced_actions(
    vector<parser::Edge>& edges,
    const unordered_set<identifier_name_t>& relevant)
{
    edges.erase(
        std::stable_partition(
            edges.begin(),
            edges.end(),
            [&](auto&& edge) {
                return !edge.action.has_value() ||
                       relevant.count(edge.action.value());
            }),
        edges.end());
}

vector<unordered_set<identifier_name_t>>
get_automata_actions(const vector<parser::Automaton>& automata)
{
    auto view = automata | std::ranges::views::transform([](auto&& automaton) {
                    return get_actions(automaton.edges);
                });
    return vector<unordered_set<identifier_name_t>>(view.begin(), view.end());
}

// using map here to obtain stable results
std::map<identifier_name_t, vector<police::size_t>> get_label_to_automata_map(
    const vector<unordered_set<identifier_name_t>>& actions)
{
    std::map<identifier_name_t, vector<police::size_t>> result;
    for (police::size_t i = 0; i < actions.size(); ++i) {
        std::for_each(actions[i].begin(), actions[i].end(), [&](auto&& action) {
            result[action].push_back(i);
        });
    }
    return result;
}

vector<Synchronization>
create_trivial_sync(const vector<parser::Automaton>& automata)
{
    const auto automata_actions = get_automata_actions(automata);
    const auto actions = get_label_to_automata_map(automata_actions);
    vector<Synchronization> syncs(actions.size());
    std::for_each(actions.begin(), actions.end(), [&](auto&& sync_action) {
        Synchronization sync;
        sync.synchronise.resize(automata.size(), std::nullopt);
        std::for_each(
            sync_action.second.begin(),
            sync_action.second.end(),
            [&](auto index) {
                sync.synchronise[index] = automata[index].name;
            });
        sync.result = sync_action.first;
        syncs.push_back(std::move(sync));
    });
    return syncs;
}

void process_system_syncs(parser::Model& model)
{
    if (model.system.syncs.has_value()) {
        for (police::size_t index = 0; index < model.automata.size(); ++index) {
            const auto all = get_actions(model.automata[index].edges);
            const auto relevant = get_relevant_actions(model.system, index);
            check_action_validity(model.automata[index].name, relevant, all);
            remove_unreferenced_actions(model.automata[index].edges, relevant);
        }
    } else {
        model.system.syncs = create_trivial_sync(model.automata);
    }
}

class SupportedExpressionChecker final : public expressions::ExpressionVisitor {
public:
    void visit(const expressions::IfThenElse&) override
    {
        POLICE_EXIT_INVALID_INPUT(
            "if-then-else expressions are currently not supported");
    }

    void visit(const expressions::NumericOperation& expr) override
    {
        switch (expr.operand) {
        case expressions::NumericOperation::Operand::MODULO: [[fallthrough]];
        case expressions::NumericOperation::Operand::DIVISION:
            POLICE_EXIT_INVALID_INPUT(
                "division expressions are currently not supported");
            break;
        case expressions::NumericOperation::Operand::MULTIPLY:
            if (!is_const(expr.left) && !is_const(expr.right)) {
                POLICE_EXIT_INVALID_INPUT(
                    "non-linear arithmetic is currently not supported");
            }
            break;
        default: break;
        }
        const bool was_numeric = numeric_;
        numeric_ = true;
        ExpressionVisitor::visit(expr);
        numeric_ = was_numeric;
    }

    void visit(const expressions::Negation& expr) override
    {
        if (numeric_) {
            POLICE_EXIT_INVALID_INPUT(
                "unsupported nesting of arithimetic and boolean operations");
        }
        expressions::ExpressionVisitor::visit(expr);
    }

    void visit(const expressions::Conjunction& expr) override
    {
        if (numeric_) {
            POLICE_EXIT_INVALID_INPUT(
                "unsupported nesting of arithimetic and boolean operations");
        }
        expressions::ExpressionVisitor::visit(expr);
    }

    void visit(const expressions::Disjunction& expr) override
    {
        if (numeric_) {
            POLICE_EXIT_INVALID_INPUT(
                "unsupported nesting of arithimetic and boolean operations");
        }
        expressions::ExpressionVisitor::visit(expr);
    }

    void visit(const expressions::BinaryFunctionCall&) override
    {
        POLICE_EXIT_INVALID_INPUT(
            "function expressions are currently not supported");
    }

    void visit(const expressions::FunctionCall&) override
    {
        POLICE_EXIT_INVALID_INPUT(
            "function expressions are currently not supported");
    }

    void visit(const expressions::Derivative&) override
    {
        POLICE_EXIT_INVALID_INPUT(
            "derivative expressions are currently not supported");
    }

private:
    bool is_const(const Expression& expr)
    {
        expr.accept(get_const_);
        return get_const_.value.has_value();
    }

    expressions::ConstantFetcher get_const_;
    bool numeric_ = false;
};

void check_supported_expressions(const parser::Model& model)
{
    SupportedExpressionChecker checker;
    visit_all(model, checker);
}

// void check_supported_properties(const vector<Property>& properties)
// {
//     if (std::any_of(properties.begin(), properties.end(), [](auto&& property)
//     {
//             const auto filter_prop =
//                 std::dynamic_pointer_cast<expressions::FilterProperty>(
//                     property.expr);
//             if (filter_prop) {
//                 const auto predicate = std::dynamic_pointer_cast<
//                     expressions::StatePredicateProperty>(filter_prop->states);
//                 const auto expr =
//                     std::dynamic_pointer_cast<expressions::StateProperty>(
//                         filter_prop->values);
//                 return predicate == nullptr || expr == nullptr ||
//                        predicate->predicate !=
//                            expressions::StatePredicateProperty::Predicate::
//                                INITIAL;
//             }
//             return true;
//         })) {
//         POLICE_EXIT_INVALID_INPUT("unsupported property");
//     }
// }

void make_final_syntactic_checks(const parser::Model& model)
{
    check_supported_expressions(model);
    // check_supported_properties(model.properties);
}

void normalize_expressions(parser::Model& model)
{
    expressions::ExpressionNormalizer normalizer;
    parser::transform_all(model, normalizer);
}

struct Scope {
    explicit Scope(
        vector<VariableDeclaration>* var_infos,
        const Scope* parent = nullptr)
        : var_infos(var_infos)
        , parent(parent)
    {
    }

    [[nodiscard]]
    std::pair<police::size_t, bool> lookup(const identifier_name_t& name) const
    {
        auto it = map.find(name);
        if (it != map.end()) return {it->second, true};
        if (parent) return parent->lookup(name);
        return {0, false};
    }

    Scope new_scope() const { return Scope(var_infos, this); }

    void define(const VariableDeclaration& var)
    {
        auto res = map.emplace(var.identifier, var_infos->size());
        if (!res.second) {
            POLICE_EXIT_INVALID_INPUT(
                "multiply defined variables with name " << var.identifier);
        }
        var_infos->push_back(var);
    }

    void define(const vector<VariableDeclaration>& declarations)
    {
        std::for_each(
            declarations.begin(),
            declarations.end(),
            [&](auto&& var) { define(var); });
    }

    unordered_map<identifier_name_t, police::size_t> map;
    vector<VariableDeclaration>* var_infos;
    const Scope* parent = nullptr;
};

class VariableRefInstantiator final
    : public expressions::ExpressionTransformer {
public:
    explicit VariableRefInstantiator(const Scope* scope)
        : scope(scope)
    {
    }

    void visit(Expression& ptr, expressions::IdentifierReference& ref)
    {
        auto var_id = scope->lookup(ref.identifier);
        if (!var_id.second) {
            POLICE_EXIT_INVALID_INPUT(
                "variable with name " << ref.identifier << " is not defined");
        }
        ptr = expressions::Variable(var_id.first);
    }

    const Scope* scope;
};

std::pair<
    unordered_map<identifier_name_t, police::size_t>,
    vector<unordered_map<identifier_name_t, police::size_t>>>
generate_variable_ids(parser::Model& model)
{
    vector<unordered_map<identifier_name_t, police::size_t>> var_maps;
    var_maps.reserve(model.automata.size());
    vector<VariableDeclaration> var_infos;
    Scope root_scope(&var_infos);
    root_scope.define(model.variables);
    for (auto& automaton : model.automata) {
        Scope local(&var_infos, &root_scope);
        local.define(automaton.variables);
        VariableRefInstantiator inst{&local};
        transform_all(automaton, inst);
        automaton.variables.clear();
        var_maps.push_back(std::move(local.map));
    }
    VariableRefInstantiator inst{&root_scope};
    transform_all(model, inst);
    model.variables.swap(var_infos);
    return {std::move(root_scope.map), std::move(var_maps)};
}

vector<police::size_t>
get_variable_reordering(const vector<VariableDeclaration>& info)
{
    vector<police::size_t> reordered_infos;
    reordered_infos.reserve(info.size());
    police::size_t reals = 0;
    police::size_t bounded_reals = 0;
    police::size_t ints = 0;
    police::size_t bounded_ints = 0;
    police::size_t bools = 0;
    police::size_t pos = 0;
    auto insert = [&](auto&& type) {
        using T = std::decay_t<decltype(type)>;
        police::size_t new_pos = 0;
        if constexpr (std::is_same_v<T, BoolType>) {
            new_pos = reordered_infos.size();
            ++bools;
        } else if constexpr (std::is_same_v<T, IntegerType>) {
            new_pos = bounded_reals + reals + ints;
            ++ints;
        } else if constexpr (std::is_same_v<T, RealType>) {
            new_pos = reals;
            ++reals;
        } else if constexpr (std::is_same_v<T, BoundedIntType>) {
            new_pos = bounded_reals + reals + ints + bounded_ints;
            ++bounded_ints;
        } else if constexpr (std::is_same_v<T, BoundedRealType>) {
            new_pos = bounded_reals + reals;
            ++bounded_reals;
        } else {
            // continuous types should have been excluded at this point
            assert(false);
        }
        reordered_infos.insert(reordered_infos.begin() + new_pos, pos);
    };
    for (pos = 0; pos < info.size(); ++pos) {
        std::visit(insert, info[pos].type);
    }
    vector<police::size_t> pos_remaps(info.size());
    for (police::size_t new_pos = 0; new_pos < pos_remaps.size(); ++new_pos) {
        pos_remaps[reordered_infos[new_pos]] = new_pos;
    }
    return pos_remaps;
}

class VariableIdReplacer final : public expressions::ExpressionTransformer {
public:
    explicit VariableIdReplacer(const vector<police::size_t>& new_ids)
        : new_ids(&new_ids)
    {
    }

    void visit(Expression&, expressions::Variable& ref)
    {
        assert(ref.var_id < new_ids->size());
        ref.var_id = new_ids->at(ref.var_id);
    }

    const vector<police::size_t>* new_ids;
};

vector<police::size_t> reorder_variable_ids(parser::Model& model)
{
    auto reordered_indices = get_variable_reordering(model.variables);
    VariableIdReplacer transformer(reordered_indices);
    transform_all(model, transformer);
    vector<VariableDeclaration> reordered_infos;
    reordered_infos.reserve(model.variables.size());
    std::for_each(
        reordered_indices.begin(),
        reordered_indices.end(),
        [&](auto old_index) {
            reordered_infos.push_back(std::move(model.variables[old_index]));
        });
    model.variables.swap(reordered_infos);
    return reordered_indices;
}

std::pair<
    unordered_map<identifier_name_t, police::size_t>,
    vector<unordered_map<identifier_name_t, police::size_t>>>
create_variable_ids(parser::Model& model)
{
    auto var_maps = generate_variable_ids(model);
    const auto new_index_map = reorder_variable_ids(model);
    auto apply_reorder = [&](auto& map) {
        for (auto it = map.begin(); it != map.end(); ++it) {
            it->second = new_index_map[it->second];
        }
    };
    auto& [global_vars, automata_vars] = var_maps;
    apply_reorder(global_vars);
    std::ranges::for_each(automata_vars, apply_reorder);
    return var_maps;
}

parser::Model parse_json(const nlohmann::json& js_object)
{
    auto model = model_schema()(js_object, true);
    if (!model.has_value()) {
        POLICE_EXIT_INVALID_INPUT(
            "An error occured while parsing the model's JSON");
    }
    return std::move(model.value());
}

struct GetOrCreateId {
    police::size_t operator()(const identifier_name_t& name) const
    {
        auto x = map->emplace(name, map->size());
        return x.first->second;
    }
    unordered_map<identifier_name_t, police::size_t>* map;
};

jani::Automaton build_automaton(
    const parser::Automaton& automaton,
    const unordered_map<identifier_name_t, police::size_t>& global_vars,
    const unordered_map<identifier_name_t, police::size_t>& automaton_vars,
    GetOrCreateId get_action_id)
{
    const unordered_map<identifier_name_t, police::size_t> location_id =
        get_index_map(
            automaton.locations,
            [](auto&& loc) { return loc.name; },
            "locations");
    auto get_var_id = [&](auto&& name) {
        auto idx = automaton_vars.find(name);
        if (idx == automaton_vars.end()) {
            idx = global_vars.find(name);
            if (idx == global_vars.end()) {
                POLICE_EXIT_INVALID_INPUT(
                    "there is no variable with name " << name);
            }
        }
        return idx->second;
    };
    auto get_loc_id = [&](auto&& name) {
        auto idx = location_id.find(name);
        if (idx == location_id.end()) {
            POLICE_EXIT_INVALID_INPUT(
                "automaton " << automaton.name << " has not location with name "
                             << name);
        }
        return idx->second;
    };
    auto translate_assignment =
        [&](const parser::Assignment& assignment) -> jani::Assignment {
        if (assignment.index != 0) {
            POLICE_EXIT_INVALID_INPUT(
                "multi-level assignments are currently not supported");
        }
        return {get_var_id(assignment.ref), assignment.value, assignment.index};
    };
    auto translate_destination = [&](const Destination& dest) -> jani::Outcome {
        auto assignments = dest.assignments |
                           std::ranges::views::transform(translate_assignment);
        return {
            get_loc_id(dest.location),
            vector<jani::Assignment>(assignments.begin(), assignments.end())};
    };
    auto initial_locations =
        automaton.initial_locations | std::ranges::views::transform(get_loc_id);
    auto location_names =
        automaton.locations |
        std::ranges::views::transform([](auto&& loc) { return loc.name; });
    auto local_vars_r =
        automaton_vars |
        std::ranges::views::transform([](auto&& pr) { return pr.second; });
    vector<police::size_t> local_vars(local_vars_r.begin(), local_vars_r.end());
    std::sort(local_vars.begin(), local_vars.end());
    jani::Automaton result(
        0,
        automaton.name,
        vector<identifier_name_t>(location_names.begin(), location_names.end()),
        {}, // labels will be computed in a later stage
        vector<police::size_t>(
            initial_locations.begin(),
            initial_locations.end()),
        std::move(local_vars));
    for (const auto& edge : automaton.edges) {
        auto outcomes = edge.destinations |
                        std::ranges::views::transform(translate_destination);
        result.add_edge(
            get_loc_id(edge.location),
            {edge.action.has_value() ? get_action_id(edge.action.value())
                                     : SILENT_ACTION,
             edge.guard.has_value()
                 ? edge.guard.value()
                 : expressions::Expression(expressions::Constant(Value(true))),
             vector<jani::Outcome>(outcomes.begin(), outcomes.end())});
    }
    return result;
}

vector<jani::Automaton> build_automata(
    const vector<parser::Automaton>& parsed_automata,
    const unordered_map<identifier_name_t, police::size_t>& global_vars,
    const vector<unordered_map<identifier_name_t, police::size_t>>&
        automata_vars,
    GetOrCreateId get_action_id)
{
    vector<jani::Automaton> automata;
    automata.reserve(parsed_automata.size());
    for (police::size_t i = 0; i < parsed_automata.size(); ++i) {
        automata.push_back(build_automaton(
            parsed_automata[i],
            global_vars,
            automata_vars[i],
            get_action_id));
        automata.back().set_id(i);
    }
    return automata;
}

vector<jani::SynchronizationVector> build_compositions(
    const vector<Synchronization>& compositions,
    GetOrCreateId get_action_id)
{
    auto build = [&](auto&& sync) {
        jani::SynchronizationVector result;
        result.label = sync.result.has_value()
                           ? get_action_id(sync.result.value())
                           : SILENT_ACTION;
        for (police::size_t idx = 0; idx < sync.synchronise.size(); ++idx) {
            if (sync.synchronise[idx].has_value()) {
                result.participants.emplace_back(
                    idx,
                    get_action_id(sync.synchronise[idx].value()));
            }
        }
        return result;
    };
    vector<jani::SynchronizationVector> result;
    result.reserve(compositions.size());
    std::transform(
        compositions.begin(),
        compositions.end(),
        std::back_inserter(result),
        build);
    return result;
}

unordered_map<size_t, size_t> make_action_ids_contiguous(
    jani::Automaton& automaton,
    const vector<identifier_name_t>& labels)
{
    unordered_map<size_t, size_t> remap;
    for (size_t loc = 0; loc < automaton.num_locations(); ++loc) {
        for (auto& edge : automaton.get_sync_edges(loc)) {
            assert(edge.action != SILENT_ACTION);
            edge.action =
                remap.emplace(edge.action, remap.size()).first->second;
        }
    }
    automaton.set_sync_labels(remap.size());
    vector<identifier_name_t> local_labels(remap.size());
    for (const auto& [i, j] : remap) {
        local_labels[j] = labels[i];
    }
    automaton.get_labels().swap(local_labels);
    return remap;
}

void make_automata_action_ids_contiguous(
    vector<jani::Automaton>& automata,
    vector<jani::SynchronizationVector>& syncs,
    const vector<identifier_name_t>& labels)
{
    vector<unordered_map<size_t, size_t>> remaps;
    remaps.reserve(automata.size());
    std::transform(
        automata.begin(),
        automata.end(),
        std::back_inserter(remaps),
        std::bind(make_action_ids_contiguous, std::placeholders::_1, labels));
    for (auto& svec : syncs) {
        for (auto& elem : svec.participants) {
            assert(elem.automaton < remaps.size());
            assert(remaps[elem.automaton].count(elem.label));
            elem.label = remaps[elem.automaton][elem.label];
        }
    }
}

Expression build_initial_state(parser::Model& parsed_model)
{
    expressions::Expression init = parsed_model.restrict_initial.has_value()
                                       ? parsed_model.restrict_initial.value()
                                       : expressions::MakeConstant()(true);
    size_t var_idx = 0;
    auto conjoin = [&](auto&& expr) {
        if (expr.has_value())
            init = init && expressions::equal(
                               expressions::Variable(var_idx - 1),
                               expr.value());
    };
    std::ranges::for_each(
        parsed_model.variables |
            std::ranges::views::transform([&](auto&& variable) {
                ++var_idx;
                return variable.initial_value;
            }),
        conjoin);
    std::ranges::for_each(
        parsed_model.automata |
            std::ranges::views::transform(
                [](auto&& automaton) { return automaton.restrict_initial; }),
        conjoin);
    expressions::ExpressionNormalizer norm;
    init.transform(norm);
    return init;
}

struct GetBoundedType {
    struct GenericType {
        Value lb;
        Value ub;

        police::BoundedRealType real() const
        {
            return {static_cast<real_t>(lb), static_cast<real_t>(ub)};
        }

        police::BoundedIntType integer() const
        {
            auto b = real();
            return {
                b.lower_bound == -std::numeric_limits<real_t>::infinity()
                    ? std::numeric_limits<int_t>::min()
                    : static_cast<int_t>(lb),
                b.upper_bound == std::numeric_limits<real_t>::infinity()
                    ? std::numeric_limits<int_t>::max()
                    : static_cast<int_t>(ub)};
        }
    };

    template <typename T>
    GenericType operator()(T&& type) const
    {
        expressions::ConstantFetcher get_const;
        Value lb(std::numeric_limits<real_t>::infinity());
        Value ub(std::numeric_limits<real_t>::infinity());
        if (type.lower_bound) {
            type.lower_bound->accept(get_const);
            if (!get_const.value.has_value()) {
                POLICE_EXIT_INVALID_INPUT(
                    "lower-bound expression of type is "
                    "not constant evaluatable");
            }
            lb = get_const.value.value();
        }
        if (type.upper_bound) {
            type.upper_bound->accept(get_const);
            if (!get_const.value.has_value()) {
                POLICE_EXIT_INVALID_INPUT(
                    "upper-bound expression of type is "
                    "not constant evaluatable");
            }
            ub = get_const.value.value();
        }
        return {lb, ub};
    }
};

VariableSpace
build_variable_structure(const vector<VariableDeclaration>& variables)
{
    VariableSpace vs;
    auto add_var = [&](auto&& type) {
        const police::size_t var_id = vs.size();
        const auto& var_name = variables[var_id].identifier;
        using T = std::decay_t<decltype(type)>;
        if constexpr (std::is_same_v<T, RealType>) {
            vs.add_variable(var_name, police::RealType{});
        } else if constexpr (std::is_same_v<T, BoundedRealType>) {
            vs.add_variable(var_name, GetBoundedType()(type).real());
        } else if constexpr (std::is_same_v<T, IntegerType>) {
            vs.add_variable(var_name, police::IntegerType{});
        } else if constexpr (std::is_same_v<T, BoundedIntType>) {
            vs.add_variable(var_name, GetBoundedType()(type).integer());
        } else if constexpr (std::is_same_v<T, BoolType>) {
            vs.add_variable(var_name, police::BoolType{});
        } else {
            assert(false);
        }
    };
    std::ranges::for_each(
        variables |
            std::ranges::views::transform([](auto&& var) { return var.type; }),
        [&](auto&& type) { std::visit(add_var, type); });
    return vs;
}

vector<identifier_name_t>
to_list(const unordered_map<identifier_name_t, police::size_t>& index_map)
{
    vector<identifier_name_t> identifiers(index_map.size());
    std::for_each(index_map.begin(), index_map.end(), [&](const auto& pr) {
        assert(pr.second < identifiers.size());
        identifiers[pr.second] = pr.first;
    });
    return identifiers;
}

struct ExtractReachaProperty {
    expressions::Expression
    operator()(const expressions::PropertyPtr& prop) const
    {
        auto filter_prop =
            std::dynamic_pointer_cast<expressions::FilterProperty>(prop);
        if (filter_prop == nullptr) {
            POLICE_EXIT_INVALID_INPUT(
                "currently only filter properties are supported");
        }
        if (filter_prop->fun != expressions::FilterProperty::Function::FORALL &&
            filter_prop->fun != expressions::FilterProperty::Function::EXISTS) {
            POLICE_EXIT_INVALID_INPUT("unsupported filter function");
        }
        auto states =
            std::dynamic_pointer_cast<expressions::StatePredicateProperty>(
                filter_prop->states);
        if (states == nullptr ||
            states->predicate !=
                expressions::StatePredicateProperty::Predicate::INITIAL) {
            POLICE_EXIT_INVALID_INPUT(
                "unsupported states expression in filter property");
        }
        auto values = std::dynamic_pointer_cast<expressions::UntilProperty>(
            filter_prop->values);
        if (values == nullptr ||
            values->op != expressions::UntilProperty::Operator::UNTIL) {
            POLICE_EXIT_INVALID_INPUT(
                "unsupported values expression in filter property");
        }
        auto left =
            std::dynamic_pointer_cast<expressions::StateProperty>(values->left);
        auto right = std::dynamic_pointer_cast<expressions::StateProperty>(
            values->right);
        if (left == nullptr || right == nullptr ||
            !left->expr.is_same(Value(true))) {
            POLICE_EXIT_INVALID_INPUT(
                "unsupported values expression in filter property");
        }
        if (filter_prop->fun == expressions::FilterProperty::Function::FORALL) {
            POLICE_WARNING("going to treat forall filter property as exists");
        }
        return right->expr;
    }
};

jani::Model build_model(parser::Model& parsed_model)
{
    unordered_map<identifier_name_t, police::size_t> action_id;
    GetOrCreateId get_action_id(&action_id);
    const auto [global_vars, automata_vars] = create_variable_ids(parsed_model);
    VariableSpace variables = build_variable_structure(parsed_model.variables);
    const auto automata_id = get_index_map(
        parsed_model.automata,
        [](auto&& automaton) { return automaton.name; },
        "automata");
    auto automata = build_automata(
        parsed_model.automata,
        global_vars,
        automata_vars,
        get_action_id);
    assert(parsed_model.system.syncs.has_value());
    auto syncs =
        build_compositions(parsed_model.system.syncs.value(), get_action_id);
    auto labels = to_list(action_id);
    make_automata_action_ids_contiguous(automata, syncs, labels);
    auto initial_state = build_initial_state(parsed_model);
    auto prop_names =
        parsed_model.properties |
        std::ranges::views::transform([](auto&& prop) { return prop.name; });
    auto prop =
        parsed_model.properties |
        std::ranges::views::transform([](auto&& prop) { return prop.expr; }) |
        std::ranges::views::transform(ExtractReachaProperty());
    auto gvr = global_vars | std::ranges::views::transform(
                                 [](auto& pt) { return pt.second; });
    vector<size_t> gv(gvr.begin(), gvr.end());
    std::sort(gv.begin(), gv.end());
    return jani::Model(
        std::move(variables),
        std::move(automata),
        std::move(syncs),
        vector<expressions::Expression>(prop.begin(), prop.end()),
        std::move(labels),
        vector<identifier_name_t>(prop_names.begin(), prop_names.end()),
        std::move(initial_state),
        std::move(gv));
}

} // namespace

police::jani::Model
parse_model(const nlohmann::json& js_object, const ConstantDefinitions& predefs)
{
    auto parsed_model = parse_json(js_object);
    make_syntactic_prechecks(parsed_model);
    process_constant_expressions(parsed_model, predefs);
    remove_unreferenced_automata(parsed_model);
    normalize_expressions(parsed_model);
    process_system_syncs(parsed_model);
    make_final_syntactic_checks(parsed_model);
    return build_model(parsed_model);
}

parser::JaniPolicyBridge parse_policy_adapter(std::string_view path)
{
    std::ifstream inf(path.data());
    auto json = nlohmann::json::parse(inf);
    return parser::jani2nn_schema()(json, true).value();
}

namespace {

vector<size_t> get_input_var_mapping(
    const vector<parser::AutomatonVariable>& input,
    const police::jani::Model& model)
{
    vector<size_t> result;
    result.reserve(input.size());
    for (auto i = 0u; i < input.size(); ++i) {
        if (input[i].automaton.has_value()) {
            auto a = std::find_if(
                model.automata.begin(),
                model.automata.end(),
                [&](auto& a) {
                    return a.get_name() == input[i].automaton.value();
                });
            if (a == model.automata.end()) {
                POLICE_EXIT_INVALID_INPUT(
                    "automaton " << input[i].automaton.value()
                                 << " does not exist");
            }
            const auto& vars = a->get_local_variables();
            auto v = std::find_if(vars.begin(), vars.end(), [&](size_t idx) {
                return model.variables[idx].name == input[i].variable;
            });
            if (v == vars.end()) {
                POLICE_EXIT_INVALID_INPUT(
                    "automaton " << input[i].automaton.value()
                                 << " does not have any variable with name "
                                 << input[i].variable);
            }
            result.push_back(*v);
        } else {
            const auto& vars = model.global_vars;
            auto v = std::find_if(vars.begin(), vars.end(), [&](size_t idx) {
                return model.variables[idx].name == input[i].variable;
            });
            if (v == vars.end()) {
                POLICE_EXIT_INVALID_INPUT(
                    "there is no global variable with name "
                    << input[i].variable);
            }
            result.push_back(*v);
        }
    }
    return result;
}

vector<size_t> get_output(
    const vector<identifier_name_t>& actions,
    const police::jani::Model& model)
{
    vector<size_t> result;
    result.reserve(actions.size());
    std::transform(
        actions.begin(),
        actions.end(),
        std::back_inserter(result),
        [&](auto&& name) {
            auto it = std::find(
                model.action_names.begin(),
                model.action_names.end(),
                name);
            if (it == model.action_names.end()) {
                POLICE_EXIT_INVALID_INPUT(
                    "there is no action with name " << name);
            }
            return std::distance(model.action_names.begin(), it);
        });
    return result;
}
} // namespace

NeuralNetworkPolicy parse_policy(
    std::string_view nnet_path,
    std::string_view bridge_path,
    const police::jani::Model& model)
{
    auto bridge = parse_policy_adapter(bridge_path);
    auto input = get_input_var_mapping(bridge.input, model);
    auto output = get_output(bridge.output, model);
    FeedForwardNeuralNetwork<> policy =
        FeedForwardNeuralNetwork<>::parse_nnet(nnet_path);
    return {
        std::move(policy),
        std::move(input),
        std::move(output),
        static_cast<size_t>(model.action_names.size())};
}

namespace {
struct VariableNameReplacer final : public expressions::ExpressionTransformer {
    void
    visit(expressions::Expression& expr, expressions::IdentifierReference& id)
        override
    {
        auto it = std::find_if(
            global_vars->begin(),
            global_vars->end(),
            [&](auto idx) { return variables->at(idx).name == id.identifier; });
        if (it == global_vars->end()) {
            POLICE_EXIT_INVALID_INPUT(
                "there is no global variable with name " << id.identifier);
        }
        expr = expressions::Variable(*it);
    }
    VariableNameReplacer(
        const vector<size_t>* global_vars,
        const VariableSpace* variables)
        : global_vars(global_vars)
        , variables(variables)
    {
    }

    const vector<size_t>* global_vars;
    const VariableSpace* variables;
};

void postprocess_expression(
    expressions::Expression& exp,
    const police::jani::Model& model)
{

    VariableNameReplacer var_ids{&model.global_vars, &model.variables};
    exp.transform(var_ids);
    expressions::ExpressionNormalizer normalizer;
    exp.transform(normalizer);
}

} // namespace

expressions::Expression parse_start_condition(
    std::string_view file_name,
    const police::jani::Model& model)
{
    std::ifstream rif(file_name.data());
    auto init = nlohmann::json::parse(rif);
    if (init["op"].get<std::string>() != "state-condition") {
        POLICE_EXIT_INVALID_INPUT("expected op='state-condition'");
    }
    if (!init.count("exp")) {
        POLICE_EXIT_INVALID_INPUT("missing exp");
    }
    if (init["exp"]["op"].get<std::string>() == "let") {
        if (init["exp"]["variables"].size() != 0) {
            POLICE_EXIT_INVALID_INPUT("expected variables = []");
        }
        init = init["exp"]["expression"];
    } else {
        init = init["exp"];
    }
    auto expr = parser::expression_schema()(init, true);
    if (!expr.has_value()) {
        POLICE_EXIT_INVALID_INPUT("failed to parse expression");
    }
    postprocess_expression(expr.value(), model);
    return expr.value();
}

namespace {
expressions::Expression _parse_state_condition(
    const nlohmann::json& init,
    const police::jani::Model& model)
{
    if (init["op"].get<std::string>() != "state-condition") {
        POLICE_EXIT_INVALID_INPUT("expected op='state-condition'");
    }
    if (!init.count("exp")) {
        POLICE_EXIT_INVALID_INPUT("missing exp");
    }
    auto expr = parser::expression_schema()(init["exp"], true);
    if (!expr.has_value()) {
        POLICE_EXIT_INVALID_INPUT("failed to parse expression");
    }
    postprocess_expression(expr.value(), model);
    return expr.value();
}
} // namespace

expressions::Expression parse_state_condition(
    std::string_view file_name,
    const police::jani::Model& model)
{
    std::ifstream rif(file_name.data());
    auto init = nlohmann::json::parse(rif);
    return _parse_state_condition(init, model);
}

#if POLICE_VERITAS

AddTreePolicy parse_addtree_policy(
    std::string_view addtree_path,
    std::string_view bridge_path,
    const police::jani::Model& model)
{
    auto bridge = parse_policy_adapter(bridge_path);
    auto input = get_input_var_mapping(bridge.input, model);
    auto output = get_output(bridge.output, model);
    std::ifstream rif(addtree_path.data());
    std::shared_ptr<veritas::AddTree> policy =
        std::make_shared<veritas::AddTree>(
            veritas::addtree_from_json<veritas::AddTree>(rif));
    return {
        std::move(policy),
        std::move(input),
        std::move(output),
        static_cast<size_t>(model.action_names.size())};
}

#else

AddTreePolicy parse_addtree_policy(
    std::string_view,
    std::string_view,
    const police::jani::Model&)
{
    POLICE_MISSING_DEPENDENCY("Veritas");
}

#endif

CGPolicy parse_cg_policy(
    std::string_view policy_path,
    std::string_view interface_path,
    const police::jani::Model& model)
{
    auto interface = parse_policy_adapter(interface_path);
    auto input = get_input_var_mapping(interface.input, model);
    auto output = get_output(interface.output, model);
    auto cg_node = cg::parse_nnet_file(policy_path);
    auto p = cg::post_process(cg_node, input, model.variables);
    return {
        std::move(p.cg),
        std::move(p.inputs),
        std::move(output),
        static_cast<size_t>(model.action_names.size())};
}

namespace {
expressions::Expression _parse_objective(
    std::string_view file_name,
    nlohmann::json prop,
    const police::jani::Model& model)
{
    if (prop.contains("file")) {
        std::filesystem::path path(file_name);
        std::ifstream rif(
            path.replace_filename(prop["file"].get<std::string>().c_str()));
        prop = nlohmann::json::parse(rif);
    }

    if (!prop.contains("goal")) {
        POLICE_EXIT_INVALID_INPUT("expected goal objective");
    }

    return _parse_state_condition(prop["goal"], model);
}

expressions::Expression _parse_start_condition(
    std::string_view file_name,
    const nlohmann::json& prop,
    const police::jani::Model& model)
{
    if (prop.contains("file")) {
        std::filesystem::path path(file_name);
        return parse_start_condition(
            path.replace_filename(prop["file"].get<std::string>()).c_str(),
            model);
    }
    return _parse_state_condition(prop, model);
}

expressions::Expression _parse_state_condition(
    std::string_view file_name,
    const nlohmann::json& prop,
    const police::jani::Model& model)
{
    if (prop.contains("file")) {
        std::filesystem::path path(file_name);
        return parse_state_condition(
            path.replace_filename(prop["file"].get<std::string>()).c_str(),
            model);
    }
    return _parse_state_condition(prop, model);
}
} // namespace

VerificationProperty parse_verification_property(
    std::string_view file_name,
    const police::jani::Model& model)
{
    std::ifstream rif(file_name.data());
    auto init = nlohmann::json::parse(rif);
    if (!init.contains("properties") || !init["properties"].is_array() ||
        init["properties"].size() != 1u) {
        POLICE_EXIT_INVALID_INPUT("properties unit-array");
    }
    auto props = init["properties"][0];
    if (!props.contains("expression")) {
        POLICE_EXIT_INVALID_INPUT("expected expression as properties element");
    }
    auto prop = props["expression"];
    if (!prop.contains("objective") || !prop.contains("reach") ||
        !prop.contains("start")) {
        POLICE_EXIT_INVALID_INPUT("expected objective, reach, and start ");
    }
    return VerificationProperty(
        _parse_start_condition(file_name, prop["start"], model),
        _parse_objective(file_name, prop["objective"], model),
        _parse_state_condition(file_name, prop["reach"], model));
}

} // namespace police::jani
