#pragma once

#include "police/addtree_policy.hpp"
#include "police/cg_policy.hpp"
#include "police/jani/model.hpp"
#include "police/jani/parser/constant_definitions.hpp"
#include "police/jani/parser/jani2nn.hpp"
#include "police/nn_policy.hpp"
#include "police/verification_property.hpp"

#include <nlohmann/json.hpp>

namespace police::jani {

/**
 * Process and normalize the Jani model returned by the parser. The given model
 * will be modified inplace (hence why creating an object copy). The processing
 * steps include (amongst other):
 * 1. Removal of automata not participating in the system's composition.
 * 2. Removal of actions not participating in the system's composition.
 * 3. Evaluation of constant expressions.
 * 4. Removal of automata edges with trivially false guard.
 * 5. Removal of actions without associated edges.
 * 6. Removal of automata compositions that are never applicable.
 * 7. Expression simplification and normalization.
 * 8. Substituting string identifiers by unsigned integers.
 **/
police::jani::Model parse_model(
    const nlohmann::json& js_object,
    const ConstantDefinitions& constants = ConstantDefinitions());

NeuralNetworkPolicy parse_policy(
    std::string_view nnet_path,
    std::string_view bridge_path,
    const police::jani::Model& model);

AddTreePolicy parse_addtree_policy(
    std::string_view addtree_path,
    std::string_view bridge_path,
    const police::jani::Model& model);

CGPolicy parse_cg_policy(
    std::string_view policy_path,
    std::string_view interface_path,
    const police::jani::Model& model);

expressions::Expression parse_start_condition(
    std::string_view file_name,
    const police::jani::Model& model);

expressions::Expression parse_state_condition(
    std::string_view file_name,
    const police::jani::Model& model);

VerificationProperty parse_verification_property(
    std::string_view file_name,
    const police::jani::Model& model);

parser::JaniPolicyBridge parse_policy_adapter(
    std::string_view file_name
);

} // namespace police::jani
