#pragma once

#include "police/base_types.hpp"
#include "police/macros.hpp"
#include "police/storage/vector.hpp"

#include <algorithm>
#include <array>
#include <initializer_list>
#include <memory>
#include <nlohmann/json.hpp>
#include <optional>
#include <string>
#include <string_view>
#include <tuple>
#include <type_traits>
#include <utility>

namespace police::jani::parser {

namespace detail {
template <typename T>
struct is_optional : public std::false_type {};

template <typename T>
struct is_optional<std::optional<T>> : public std::true_type {};

template <typename T>
constexpr bool is_optional_v = is_optional<T>::value;
} // namespace detail

template <typename T>
concept JaniProcessor = requires(T processor) {
    processor(std::declval<nlohmann::json>(), bool{});
    requires detail::is_optional_v<
        decltype(processor(std::declval<nlohmann::json>(), bool{}))>;
};

#define PLAIN_OBJECT(ClassName, Validation, ErrorMessage, TypeName)            \
    template <typename ValueFactory>                                           \
    struct ClassName {                                                         \
        using ValueType = decltype(ValueFactory()(std::declval<TypeName>()));  \
        ValueFactory factory;                                                  \
        explicit ClassName(ValueFactory factory = ValueFactory())              \
            : factory(std::move(factory))                                      \
        {                                                                      \
        }                                                                      \
        constexpr std::optional<ValueType>                                     \
        operator()(const nlohmann::json& js, bool report) const                \
        {                                                                      \
            if (!js.Validation()) {                                            \
                if (report)                                                    \
                    POLICE_PRINT_ERROR(                                        \
                        "failed to parse " << js << ": " << ErrorMessage);     \
                return std::nullopt;                                           \
            }                                                                  \
            return {factory(js.get<TypeName>())};                              \
        }                                                                      \
    };

PLAIN_OBJECT(JaniString, is_string, "is not a string", std::string)
PLAIN_OBJECT(JaniInteger, is_number_integer, "is not an integer", int)
PLAIN_OBJECT(JaniReal, is_number, "is not a number", real_t)
PLAIN_OBJECT(JaniBool, is_boolean, "is not a Boolean", bool)

#undef PLAIN_OBJECT

namespace detail {
template <JaniProcessor Processor>
struct value_type {
private:
    using optional_value_type = decltype(std::declval<Processor>()(
        std::declval<nlohmann::json>(),
        bool{}));

public:
    using type = typename optional_value_type::value_type;
};

template <JaniProcessor Processor>
using value_type_t = typename value_type<Processor>::type;

template <typename ValueType>
class AbstractProcessor {
public:
    virtual ~AbstractProcessor() = default;
    virtual std::optional<ValueType>
    parse(const nlohmann::json& js, bool report) const = 0;
};

template <JaniProcessor Processor, typename ValueType = value_type_t<Processor>>
class ProcessorInstance final : public AbstractProcessor<ValueType> {
public:
    constexpr explicit ProcessorInstance(Processor processor)
        : proc_(std::move(processor))
    {
    }

    constexpr std::optional<ValueType>
    parse(const nlohmann::json& js, bool report) const override
    {
        auto result = proc_(js, report);
        if constexpr (std::is_same_v<ValueType, value_type_t<Processor>>) {
            return result;
        } else if (result.has_value()) {
            ValueType vt{std::move(result.value())};
            return {std::move(vt)};
        } else {
            return std::nullopt;
        }
    }

private:
    Processor proc_;
};

template <typename ValueType, JaniProcessor Processor>
std::shared_ptr<AbstractProcessor<ValueType>>
make_instance(Processor&& processor)
{
    return std::make_shared<
        ProcessorInstance<std::remove_cvref_t<Processor>, ValueType>>(
        std::forward<Processor>(processor));
}

template <JaniProcessor Processor>
std::shared_ptr<AbstractProcessor<value_type_t<Processor>>>
make_instance_dynamic(Processor&& processor)
{
    return make_instance<value_type_t<Processor>>(
        std::forward<Processor>(processor));
}

} // namespace detail

template <typename ValueType>
struct JaniSchema {
    constexpr JaniSchema()
        : fref_(
              std::make_shared<
                  std::unique_ptr<detail::AbstractProcessor<ValueType>>>(
                  nullptr))
    {
    }

    template <
        JaniProcessor Processor,
        std::enable_if_t<
            !std::is_same_v<std::remove_cvref_t<Processor>, JaniSchema>,
            bool> = true>
    constexpr explicit JaniSchema(Processor&& processor)
        : fref_(
              std::make_shared<
                  std::unique_ptr<detail::AbstractProcessor<ValueType>>>(
                  std::make_unique<detail::ProcessorInstance<
                      std::remove_cvref_t<Processor>>>(
                      std::forward<Processor>(processor))))
    {
        static_assert(
            std::is_same_v<ValueType, detail::value_type_t<Processor>>);
    }

    template <
        JaniProcessor Processor,
        std::enable_if_t<
            !std::is_same_v<std::remove_cvref_t<Processor>, JaniSchema>,
            bool> = true>
    constexpr JaniSchema& operator=(Processor&& processor)
    {
        *fref_ = std::make_unique<
            detail::ProcessorInstance<std::remove_cvref_t<Processor>>>(
            std::forward<Processor>(processor));
        return *this;
    }

    constexpr std::optional<ValueType>
    operator()(const nlohmann::json& js, bool report) const
    {
        assert(fref_ != nullptr && *fref_ != nullptr);
        return (*fref_)->parse(js, report);
    }

private:
    std::shared_ptr<std::unique_ptr<detail::AbstractProcessor<ValueType>>>
        fref_;
};

template <JaniProcessor Processor>
JaniSchema(Processor&&) -> JaniSchema<detail::value_type_t<Processor>>;

template <typename ValueType>
struct JaniList {
    static_assert(std::is_move_constructible_v<ValueType>);

    template <
        JaniProcessor Processor,
        std::enable_if_t<
            !std::
                is_same_v<std::remove_cvref_t<Processor>, JaniList<ValueType>>,
            bool> = true>
    constexpr explicit JaniList(Processor&& processor)
        : proc_(
              detail::make_instance<ValueType>(
                  std::forward<Processor>(processor)))
    {
    }

    std::optional<vector<ValueType>>
    operator()(const nlohmann::json& js, bool report) const
    {
        if (!js.is_array()) {
            if (report)
                POLICE_PRINT_ERROR(
                    "failed to parse " << js << ": not an array");
            return std::nullopt;
        }
        vector<ValueType> list;
        for (auto it = js.cbegin(); it != js.cend(); ++it) {
            std::optional<ValueType> value = proc_->parse(*it, report);
            if (!value.has_value()) {
                if (report) POLICE_PRINT_ERROR("aborting parsing list " << js);
                return std::nullopt;
            }
            list.push_back(std::move(value.value()));
        }
        return {std::move(list)};
    }

private:
    std::shared_ptr<detail::AbstractProcessor<ValueType>> proc_;
};

template <JaniProcessor Processor>
JaniList(Processor&&) -> JaniList<detail::value_type_t<Processor>>;

namespace detail {
struct DictionaryParsingState {
    unsigned skipped = 0;
    unsigned parsed = 0;
    bool valid = true;
};
} // namespace detail

template <typename ValueType>
class JaniDictArgument {
public:
    static_assert(
        std::is_constructible_v<ValueType> &&
        std::is_move_constructible_v<ValueType>);

    using value_type_t = ValueType;

    template <JaniProcessor Processor>
    constexpr JaniDictArgument(
        std::string_view key,
        Processor&& processor,
        bool optional = false)
        : key_(key)
        , proc_(
              detail::make_instance<ValueType>(
                  std::forward<Processor>(processor)))
        , optional_(optional)
    {
    }

    constexpr ValueType operator()(
        const nlohmann::json& js,
        bool report,
        detail::DictionaryParsingState& state) const
    {
        if (!state.valid) return {};
        if (js.contains(key_)) {
            auto val = proc_->parse(js[key_], report);
            if (val.has_value()) {
                ++state.parsed;
                return std::move(val.value());
            } else {
                if (report)
                    POLICE_PRINT_ERROR("aborting parsing object " << js);
                state.valid = false;
            }
        } else if (!optional_) {
            state.valid = false;
            if (report)
                POLICE_PRINT_ERROR(
                    "failed to parse object "
                    << js << ": does not contain required entry with key "
                    << key_);
        } else {
            ++state.skipped;
        }
        return {};
    }

private:
    std::string key_;
    std::shared_ptr<detail::AbstractProcessor<ValueType>> proc_;
    bool optional_;
};

template <JaniProcessor Processor>
JaniDictArgument(std::string_view, Processor&&)
    -> JaniDictArgument<detail::value_type_t<Processor>>;

template <JaniProcessor Processor>
JaniDictArgument(std::string_view, Processor&&, bool)
    -> JaniDictArgument<detail::value_type_t<Processor>>;

class JaniDictElement {
public:
    template <JaniProcessor Processor>
    JaniDictElement(
        std::string_view key,
        Processor&& processor,
        bool optional = false)
        : key_(key)
        , test_(
              std::make_shared<BaseImpl<Processor>>(
                  std::forward<Processor>(processor)))
        , optional_(optional)
    {
    }

    constexpr bool operator()(
        const nlohmann::json& js,
        bool report,
        detail::DictionaryParsingState& state) const
    {
        if (!state.valid) return false;
        if (js.contains(key_)) {
            state.valid = test_->parse(js[key_], report);
            state.parsed += state.valid;
        } else if (!optional_) {
            state.valid = false;
            if (report)
                POLICE_PRINT_ERROR(
                    "failed to parse object "
                    << js << ": does not contain required entry with key "
                    << key_);
        } else {
            ++state.skipped;
        }
        return state.valid;
    }

    struct Base {
        virtual ~Base() = default;
        virtual bool parse(const nlohmann::json& js, bool report) const = 0;
    };

    template <JaniProcessor Processor>
    struct BaseImpl final : public Base {
        explicit BaseImpl(Processor processor)
            : processor(std::move(processor))
        {
        }
        constexpr bool
        parse(const nlohmann::json& js, bool report) const override
        {
            auto val = processor(js, report);
            return val.has_value();
        }
        Processor processor;
    };

private:
    std::string key_;
    std::shared_ptr<Base> test_;
    bool optional_;
};

template <typename ValueFactory, typename... ArgumentTypes>
struct JaniDictionary {
    using ValueType = decltype(std::declval<ValueFactory>()(
        std::declval<ArgumentTypes>()...));

    template <
        typename... Arguments,
        std::enable_if_t<
            sizeof...(Arguments) == sizeof...(ArgumentTypes) &&
                sizeof...(Arguments) == 0,
            bool> = true>
    constexpr JaniDictionary(
        ValueFactory factory,
        std::initializer_list<JaniDictElement> elements,
        Arguments&&...)
        : factory_(std::move(factory))
        , elements_(elements)
    {
    }

    template <
        typename Argument,
        typename... Arguments,
        std::enable_if_t<
            !std::is_same_v<std::remove_cvref_t<Argument>, JaniDictionary>,
            bool> = true>
    constexpr JaniDictionary(
        ValueFactory factory,
        std::initializer_list<JaniDictElement> elements,
        Argument&& arg0,
        Arguments&&... args)
        : factory_(std::move(factory))
        , args_(
              {std::forward<Argument>(arg0), std::forward<Arguments>(args)...})
        , elements_(elements)
    {
        static_assert(sizeof...(args) + 1 == sizeof...(ArgumentTypes));
    }

    constexpr std::optional<ValueType>
    operator()(const nlohmann::json& js, bool report) const
    {
        if (!js.is_object()) {
            if (report)
                POLICE_PRINT_ERROR(
                    "failed to parse " << js << ": not a json object");
            return std::nullopt;
        }
        detail::DictionaryParsingState state;
        std::for_each(
            elements_.begin(),
            elements_.end(),
            [&state, &js, &report](const auto& elem) {
                elem(js, report, state);
            });
        if (!state.valid) {
            if (report) POLICE_PRINT_ERROR("failed to parse dict " << js);
            return std::nullopt;
        }
        auto process_args = [&](auto&&... args) -> std::optional<ValueType> {
            auto tupl = std::tuple(args(js, report, state)...);
            if (state.valid) {
                return std::apply(factory_, tupl);
            }
            return std::nullopt;
        };
        auto value = std::apply(process_args, args_);
        if (!state.valid) {
            if (report) POLICE_PRINT_ERROR("failed to parse dict " << js);
            return std::nullopt;
        } else if (js.size() != state.parsed) {
            if (report)
                POLICE_PRINT_ERROR(
                    "failed to parse dict " << js << ": unrecognized elements");
            return std::nullopt;
        }
        return value;
    }

private:
    ValueFactory factory_;
    std::tuple<JaniDictArgument<ArgumentTypes>...> args_;
    vector<JaniDictElement> elements_;
};

template <typename ValueFactory, typename... Arguments>
JaniDictionary<ValueFactory, typename Arguments::value_type_t...>
make_dictionary(
    ValueFactory factory,
    std::initializer_list<JaniDictElement> elements,
    Arguments&&... args)
{
    return {
        std::move(factory),
        std::move(elements),
        std::forward<Arguments>(args)...};
}

template <typename ValueFactory, const char*... Values>
struct JaniEnum {
    using ValueType = decltype(std::declval<ValueFactory>()(int{}));

    constexpr static std::array<const char*, sizeof...(Values)> options = {
        Values...};

    ValueFactory factory_;

    explicit JaniEnum(ValueFactory factory = ValueFactory())
        : factory_(std::move(factory))
    {
    }

    constexpr std::optional<ValueType>
    operator()(const nlohmann::json& js, bool report) const
    {
        if (!js.is_string()) {
            if (report)
                POLICE_PRINT_ERROR(
                    "failed to parse enum " << js << ": is not a string");
            return std::nullopt;
        }
        const std::string str = js.get<std::string>();
        for (auto i = 0u; i < options.size(); ++i) {
            if (str == options[i]) {
                return {factory_(i)};
            }
        }
        if (report)
            POLICE_PRINT_ERROR(
                "failed to parse enum " << js << ": no matching enum value");
        return std::nullopt;
    }
};

template <typename ValueType>
struct JaniMultiSchemata {
    template <
        JaniProcessor Processor0,
        JaniProcessor... Processors,
        std::enable_if_t<
            !std::is_same_v<std::remove_cvref_t<JaniMultiSchemata>, Processor0>,
            bool> = true>
    constexpr JaniMultiSchemata(
        Processor0 processor0,
        Processors&&... processors)
        : factories_(
              {detail::make_instance_dynamic(
                   std::forward<Processor0>(processor0)),
               detail::make_instance_dynamic(
                   std::forward<Processors>(processors))...})
    {
        static_assert(sizeof...(Processors) > 0);
        static_assert(
            std::is_same_v<detail::value_type_t<Processor0>, ValueType> &&
            (... &&
             std::is_same_v<detail::value_type_t<Processors>, ValueType>));
    }

    constexpr std::optional<ValueType>
    operator()(const nlohmann::json& js, bool report) const
    {
        for (auto i = 0u; i < factories_.size(); ++i) {
            auto value = factories_[i]->parse(js, false);
            if (value.has_value()) {
                return value;
            }
        }
        if (report)
            POLICE_PRINT_ERROR(
                "failed to parse " << js << ": no matching option");
        return std::nullopt;
    }

private:
    vector<std::shared_ptr<detail::AbstractProcessor<ValueType>>> factories_;
};

template <JaniProcessor Processor, JaniProcessor... Processors>
JaniMultiSchemata(Processor&&, Processors&&...)
    -> JaniMultiSchemata<detail::value_type_t<Processor>>;

template <typename ValueType>
struct JaniNull {
    template <
        JaniProcessor Processor,
        std::enable_if_t<
            !std::is_same_v<std::remove_cvref_t<Processor>, JaniNull>,
            bool> = true>
    constexpr explicit JaniNull(Processor&& processor)
        : proc_(
              detail::make_instance<ValueType>(
                  std::forward<Processor>(processor)))
    {
        static_assert(
            std::is_same_v<ValueType, detail::value_type_t<Processor>>);
    }

    constexpr std::optional<std::optional<ValueType>>
    operator()(const nlohmann::json& js, bool report) const
    {
        if (js.is_null()) {
            std::optional<ValueType> opt = std::nullopt;
            return {opt};
        }
        auto val = proc_->parse(js, report);
        if (!val.has_value()) {
            return std::nullopt;
        }
        return {std::move(val)};
    }

private:
    std::shared_ptr<detail::AbstractProcessor<ValueType>> proc_;
};

template <JaniProcessor Processor>
JaniNull(Processor&&) -> JaniNull<detail::value_type_t<Processor>>;

template <typename ValueFactory, typename OriginalValueType>
struct JaniCast {
    using ValueType = decltype(std::declval<ValueFactory>()(
        std::declval<OriginalValueType>()));

    template <
        JaniProcessor Processor,
        std::enable_if_t<
            !std::is_same_v<std::remove_cvref_t<Processor>, JaniCast>,
            bool> = true>
    constexpr explicit JaniCast(ValueFactory factory, Processor&& processor)
        : proc_(
              detail::make_instance_dynamic(std::forward<Processor>(processor)))
        , factory_(std::move(factory))
    {
    }

    constexpr std::optional<ValueType>
    operator()(const nlohmann::json& js, bool report) const
    {
        auto val = proc_->parse(js, report);
        if (!val.has_value()) {
            return std::nullopt;
        }
        return {factory_(val.value())};
    }

private:
    std::shared_ptr<detail::AbstractProcessor<OriginalValueType>> proc_;
    ValueFactory factory_;
};

template <typename ValueFactory, JaniProcessor Processor>
JaniCast<ValueFactory, detail::value_type_t<Processor>>
make_cast(ValueFactory&& factory, Processor&& processor)
{
    return JaniCast<ValueFactory, detail::value_type_t<Processor>>(
        std::forward<ValueFactory>(factory),
        std::forward<Processor>(processor));
}

namespace factories {
template <typename T>
struct Construct {
    template <typename... Args>
    constexpr T operator()(Args&&... args) const
    {
        return T(std::forward<Args>(args)...);
    }
};

template <typename T>
struct OptionalConstruct {
    template <typename... Args>
    constexpr std::optional<T> operator()(Args&&... args) const
    {
        return {T(std::forward<Args>(args)...)};
    }
};

template <typename BaseType, typename ActualType>
struct MakeShared {
    template <typename... Args>
    constexpr std::shared_ptr<BaseType> operator()(Args&&... args) const
    {
        return std::make_shared<ActualType>(std::forward<Args>(args)...);
    }
};

template <typename CastType>
struct CastTo {
    template <typename CastFrom>
    constexpr CastType operator()(CastFrom&& src) const
    {
        return CastType(src);
    }
};

struct Discard {
    template <typename T>
    constexpr bool operator()(T&&) const
    {
        return false;
    }
};
} // namespace factories

} // namespace police::jani::parser
