#include "police/jani/parser/types.hpp"
#include "police/jani/parser/expression.hpp"
#include "police/jani/parser/language.hpp"
#include "police/jani/parser/schema_factory.hpp"
#include "police/macros.hpp"

namespace police::jani::parser {

namespace {

struct BasicTypeFactory {
    POLICE_CONSTEXPR_VARIANT Type operator()(int t) const
    {
        switch (t) {
        case 0: return BoolType{};
        case 1: return IntegerType{};
        case 2: return RealType{};
        }
        POLICE_UNREACHABLE();
    }
};

struct ContinuousTypeFactory {
    POLICE_CONSTEXPR_VARIANT Type operator()(int t) const
    {
        switch (t) {
        case 0: return ClockType{};
        case 1: return ContinuousType{};
        }
        POLICE_UNREACHABLE();
    }
};

struct BoundedTypeFactory {
    Type operator()(
        int t,
        std::optional<Expression> left,
        std::optional<Expression> right) const
    {
        switch (t) {
        case 0: return BoundedIntType{std::move(left), std::move(right)};
        case 1: return BoundedRealType{std::move(left), std::move(right)};
        }
        POLICE_UNREACHABLE();
    }
};

} // namespace

JaniSchema<Type> basic_type_schema()
{
    static auto schema = JaniSchema(
        JaniEnum<BasicTypeFactory, lang::BOOL, lang::INT, lang::REAL>());
    return schema;
}

JaniSchema<Type> bounded_type_schema()
{
    static auto schema = JaniSchema(make_dictionary<BoundedTypeFactory>(
        BoundedTypeFactory(),
        {JaniDictElement(
            lang::KIND,
            JaniEnum<factories::Discard, lang::BOUNDED>())},
        JaniDictArgument(
            lang::BASE,
            JaniEnum<factories::Construct<int>, lang::INT, lang::REAL>()),
        JaniDictArgument<std::optional<Expression>>(
            lang::LOWER_BOUND,
            expression_schema(),
            true),
        JaniDictArgument<std::optional<Expression>>(
            lang::UPPER_BOUND,
            expression_schema(),
            true)));
    return schema;
}

JaniSchema<Type> type_schema()
{
    static auto schema = JaniSchema(JaniMultiSchemata(
        basic_type_schema(),
        bounded_type_schema(),
        JaniEnum<ContinuousTypeFactory, lang::CLOCK, lang::CONTINUOUS>()));
    return schema;
}

} // namespace police::jani::parser
