#pragma once

#include "police/base_types.hpp"
#include "police/expressions/expression.hpp"
#include "police/linear_condition.hpp"
#include "police/linear_constraint.hpp"
#include "police/sat_model.hpp"
#include "police/storage/vector.hpp"

#include <limits>
#include <string_view>

namespace police {

struct LPVariable {
    enum class Type { BOOL, INT, REAL };

    explicit LPVariable(
        real_t lower_bound = -police::infinity<real_t>,
        real_t upper_bound = police::infinity<real_t>,
        real_t obj_coef = 0.,
        Type type = Type::REAL,
        std::string_view name = "");

    std::string name;
    real_t lower_bound;
    real_t upper_bound;
    real_t obj_coef;
    Type type;
};

struct IndicatorLPConstraint {
    IndicatorLPConstraint(
        size_t indicator_var,
        bool indicator_value,
        LinearConstraint constraint);
    size_t indicator_var;
    bool indicator_value;
    LinearConstraint constraint;
};

// y = max(elements..., c)
struct MaxConstraint {
    vector<size_t> elements;
    size_t y;
    real_t c = -std::numeric_limits<real_t>::infinity();

    [[nodiscard]]
    expressions::Expression as_expression() const;
};

enum LPOptimizationKind { MINIMIZE, MAXIMIZE };

enum LPStatus {
    OPTIMAL = 0,
    SOLVABLE = 1,
    INFEASIBLE = 2,
    UNBOUNDED = 3,
};

class LP {
public:
    using variable_ref = size_t;
    using constraint_ref = size_t;
    using variable_type = LPVariable;
    using linear_constraint_type = LinearConstraint;
    using disjunctive_constraint_type = LinearConstraintDisjunction;
    using indicator_constraint_type = IndicatorLPConstraint;
    using optimization_kind = LPOptimizationKind;
    using max_constraint_type = MaxConstraint;
    using model_type = SATModel;

    explicit LP(LPOptimizationKind opt_kind);

    virtual ~LP() = default;

    virtual void set_sense(LPOptimizationKind sense) = 0;

    virtual variable_ref add_variable(const variable_type& var) = 0;

    virtual constraint_ref
    add_constraint(const linear_constraint_type& constraint) = 0;

    virtual constraint_ref
    add_constraint(const disjunctive_constraint_type& constraint);

    virtual constraint_ref
    add_constraint(const indicator_constraint_type& constraint);

    virtual constraint_ref
    add_constraint(const max_constraint_type& constraint);

    virtual void set_variable_lower_bound(variable_ref var, real_t lb) = 0;

    virtual void set_variable_upper_bound(variable_ref var, real_t lb) = 0;

    [[nodiscard]]
    virtual size_t num_constraints() const = 0;

    [[nodiscard]]
    virtual size_t num_variables() const = 0;

    [[nodiscard]]
    virtual optimization_kind get_sense() const = 0;

    [[nodiscard]]
    virtual LPStatus solve() = 0;

    [[nodiscard]]
    virtual LPStatus
    solve(const vector<linear_constraint_type>& assumptions) = 0;

    [[nodiscard]]
    virtual vector<linear_constraint_type> get_unsat_core() const;

    virtual void push_snapshot() = 0;

    virtual void pop_snapshot() = 0;

    [[nodiscard]]
    virtual model_type get_model() const = 0;

    [[nodiscard]]
    virtual real_t get_objective_value() const = 0;

    [[nodiscard]]
    virtual std::string get_name() const = 0;

    [[nodiscard]]
    virtual real_t get_infinity() const = 0;

    virtual void dump(std::ostream& out) const;

    virtual void dump_model(std::ostream& out) const;
};

} // namespace police
