#pragma once

#include "police/base_types.hpp"
#include "police/lp.hpp"
#include "police/lp_factory.hpp"
#include "police/storage/id_map.hpp"
#include "police/storage/variable_space.hpp"

#include <memory>

class GRBVar;
class GRBConstr;
class GRBGenConstr;
class GRBSOS;

namespace police {

class GurobiLP final : public LP {
public:
    GurobiLP(LPOptimizationKind sense);

    [[nodiscard]]
    model_type get_model() const override;

    [[nodiscard]]
    real_t get_objective_value() const override;

    [[nodiscard]]
    std::string get_name() const override;

    [[nodiscard]]
    real_t get_infinity() const override;

    void push_snapshot() override;

    void pop_snapshot() override;

    [[nodiscard]]
    size_t num_constraints() const override;

    [[nodiscard]]
    size_t num_variables() const override;

    [[nodiscard]]
    LPStatus solve() override;

    [[nodiscard]]
    LPStatus solve(const vector<linear_constraint_type>& assumptions) override;

    [[nodiscard]]
    vector<linear_constraint_type> get_unsat_core() const override;

    variable_ref add_variable(const variable_type& var) override;

    constraint_ref
    add_constraint(const linear_constraint_type& constraint) override;

    constraint_ref
    add_constraint(const disjunctive_constraint_type& constraint) override;

    constraint_ref
    add_constraint(const indicator_constraint_type& constraint) override;

    constraint_ref
    add_constraint(const max_constraint_type& constraint) override;

    void set_variable_lower_bound(variable_ref var, real_t lb) override;

    void set_variable_upper_bound(variable_ref var, real_t lb) override;

    void set_sense(LPOptimizationKind sense) override;

    [[nodiscard]]
    LPOptimizationKind get_sense() const override;

    void dump(std::ostream& out) const override;

    void dump_model(std::ostream& out) const override;

private:
    struct GurobiInternals;

    struct GRBDeleter {
        void operator()(GurobiInternals* model) const;
    };

#ifdef POLICE_GUROBI
    struct snapshot {
        size_t vars{};
        size_t total_vars{};
        size_t lin_constraints{};
        size_t gen_constraints{};
        size_t indicator_constraints{};
    };

    [[nodiscard]]
    size_t
    get_indicator_constraint_index(const linear_constraint_type& constraint);

    [[nodiscard]]
    variable_ref
    create_indicator_constraint(const linear_constraint_type& constraint);

    void add_indicator_constraint(
        size_t var,
        bool val,
        const linear_constraint_type& constraint);

    void bind_indicator_value(size_t var, bool val, size_t indicator);

    [[nodiscard]]
    size_t get_num_lin_constraints() const;

    [[nodiscard]]
    size_t get_num_gen_constraints() const;

    [[nodiscard]]
    size_t get_num_variables() const;

    void notify_new_variable(GRBVar& var);
    void notify_new_constraint(GRBConstr& var);
    void notify_new_constraint(GRBGenConstr& var);
    void notify_new_constraint(GRBSOS& var);

    int clear_objective();
    void restore_objective(int sense);

    size_t num_vars_ = 0;
    VariableSpace vspace_;
    vector<size_t> var_remap_;
    vector<real_t> objective_coefs_;
    vector<size_t> linear_constraints_indices_;
    vector<snapshot> snapshots_;
    id_map<LinearConstraint> indicator_constraints_;
    vector<std::pair<size_t, bool>> indicator_vars_;
    vector<linear_constraint_type> unsat_core_;
    std::unique_ptr<GurobiInternals, GRBDeleter> gurobi_;
#endif
};

class GurobiLPFactory : public LPFactory {
private:
    [[nodiscard]]
    LP* make(LPOptimizationKind kind) const
    {
        return new GurobiLP(kind);
    }
};

} // namespace police
