#include "police/lp_z3.hpp"

#if POLICE_Z3

#include "police/_bits/z3_optimizer.hpp"
#include "police/lp.hpp"
#include "police/macros.hpp"
#include "police/option.hpp"
#include "police/storage/variable_space.hpp"

#include <limits>
#include <memory>

namespace police {

Z3LP::Z3LP(LPOptimizationKind opt)
    : LP(opt)
    , sense_(opt)
    , optimizer_(std::make_unique<Z3Optimizer>())
{
}

void Z3LP::set_sense(LPOptimizationKind sense)
{
    sense_ = sense;
}

LP::variable_ref Z3LP::add_variable(const variable_type& var)
{
    clear_optimizer_state();
    if (var.type == variable_type::Type::REAL) {
        optimizer_->add_variable(RealType());
    } else {
        optimizer_->add_variable(IntegerType());
    }
    var_lbs_.push_back(var.lower_bound);
    var_ubs_.push_back(var.upper_bound);
    objective_coefs_.push_back(var.obj_coef);
    return optimizer_->get_environment().num_variables() - 1;
}

LP::constraint_ref
Z3LP::add_constraint(const linear_constraint_type& constraint)
{
    clear_optimizer_state();
    optimizer_->add_constraint(constraint);
    return -1;
}

LP::constraint_ref
Z3LP::add_constraint(const disjunctive_constraint_type& constraint)
{
    clear_optimizer_state();
    optimizer_->add_constraint(constraint);
    return -1;
}

LP::constraint_ref
Z3LP::add_constraint(const indicator_constraint_type& constraint)
{
    clear_optimizer_state();
    optimizer_->add_constraint(constraint);
    return -1;
}

LP::constraint_ref Z3LP::add_constraint(const max_constraint_type& constraint)
{
    clear_optimizer_state();
    optimizer_->add_constraint(constraint);
    return -1;
}

void Z3LP::set_variable_lower_bound(variable_ref, real_t)
{
    POLICE_RUNTIME_ERROR(
        "z3lp currently doesn't support changing variable bounds");
}

void Z3LP::set_variable_upper_bound(variable_ref, real_t)
{
    POLICE_RUNTIME_ERROR(
        "z3lp currently doesn't support changing variable bounds");
}

size_t Z3LP::num_constraints() const
{
    return -1;
}

size_t Z3LP::num_variables() const
{
    return optimizer_->get_environment().num_variables();
}

LP::optimization_kind Z3LP::get_sense() const
{
    return sense_;
}

void Z3LP::set_variable_bounds()
{
    for (size_t var = 0; var < num_variables(); ++var) {
        if (var_lbs_[var] != -std::numeric_limits<real_t>::infinity()) {
            LinearConstraint c(LinearConstraint::GREATER_EQUAL);
            c.insert(var, 1.);
            c.rhs = var_lbs_[var];
            optimizer_->add_constraint(c);
        }
        if (var_ubs_[var] != std::numeric_limits<real_t>::infinity()) {
            LinearConstraint c(LinearConstraint::LESS_EQUAL);
            c.insert(var, 1.);
            c.rhs = var_ubs_[var];
            optimizer_->add_constraint(c);
        }
    }
}

void Z3LP::add_objective()
{
    LinearExpression expr;
    for (size_t var = 0; var < num_variables(); ++var) {
        if (objective_coefs_[var] != 0.) {
            expr.insert(var, objective_coefs_[var]);
        }
    }
    if (!expr.empty()) {
        if (sense_ == LPOptimizationKind::MAXIMIZE) {
            optimizer_->add_max_objective(expr);
        } else {
            optimizer_->add_min_objective(expr);
        }
    }
}

LPStatus Z3LP::solve()
{
    return solve({});
}

LPStatus Z3LP::solve(const vector<linear_constraint_type>& assumptions)
{
    clear_optimizer_state();
    optimizer_->push_snapshot();
    set_variable_bounds();
    add_objective();
    dirty_ = true;
    if (optimizer_->check(assumptions)) {
        return LPStatus::OPTIMAL;
    }
    return LPStatus::INFEASIBLE;
}

vector<LP::linear_constraint_type> Z3LP::get_unsat_core() const
{
    return optimizer_->get_unsat_core();
}

void Z3LP::push_snapshot()
{
    clear_optimizer_state();
    optimizer_->push_snapshot();
}

void Z3LP::pop_snapshot()
{
    clear_optimizer_state();
    optimizer_->pop_snapshot();
}

LP::model_type Z3LP::get_model() const
{
    return optimizer_->get_model();
}

real_t Z3LP::get_objective_value() const
{
    return optimizer_->get_objective_value();
}

std::string Z3LP::get_name() const
{
    return "z3";
}

real_t Z3LP::get_infinity() const
{
    return std::numeric_limits<real_t>::infinity();
}

void Z3LP::dump(std::ostream& out) const
{
    optimizer_->dump(out);
}

void Z3LP::dump_model(std::ostream&) const
{
}

void Z3LP::clear_optimizer_state()
{
    if (dirty_) {
        dirty_ = false;
        pop_snapshot();
    }
}

namespace {
PointerOption<LPFactory> _option("z3", [](const Arguments&) {
    return std::make_shared<Z3LPFactory>();
});
} // namespace

} // namespace police

#else

namespace police {

Z3LP::Z3LP(LPOptimizationKind opt)
    : LP(opt)
{
}

void Z3LP::set_sense(LPOptimizationKind)
{
    POLICE_MISSING_DEPENDENCY("z3");
}

LP::variable_ref Z3LP::add_variable(const variable_type&)
{
    POLICE_MISSING_DEPENDENCY("z3");
}

LP::constraint_ref Z3LP::add_constraint(const linear_constraint_type&)
{
    POLICE_MISSING_DEPENDENCY("z3");
}

LP::constraint_ref Z3LP::add_constraint(const disjunctive_constraint_type&)
{
    POLICE_MISSING_DEPENDENCY("z3");
}

LP::constraint_ref Z3LP::add_constraint(const indicator_constraint_type&)
{
    POLICE_MISSING_DEPENDENCY("z3");
}

LP::constraint_ref Z3LP::add_constraint(const max_constraint_type&)
{
    POLICE_MISSING_DEPENDENCY("z3");
}

void Z3LP::set_variable_lower_bound(variable_ref, real_t)
{
    POLICE_MISSING_DEPENDENCY("z3");
}

void Z3LP::set_variable_upper_bound(variable_ref, real_t)
{
    POLICE_MISSING_DEPENDENCY("z3");
}

size_t Z3LP::num_constraints() const
{
    POLICE_MISSING_DEPENDENCY("z3");
}

size_t Z3LP::num_variables() const
{
    POLICE_MISSING_DEPENDENCY("z3");
}

LP::optimization_kind Z3LP::get_sense() const
{
    POLICE_MISSING_DEPENDENCY("z3");
}

LPStatus Z3LP::solve()
{
    POLICE_MISSING_DEPENDENCY("z3");
}

LPStatus Z3LP::solve(const vector<linear_constraint_type>&)
{
    POLICE_MISSING_DEPENDENCY("z3");
}

vector<LP::linear_constraint_type> Z3LP::get_unsat_core() const
{
    POLICE_MISSING_DEPENDENCY("z3");
}

void Z3LP::push_snapshot()
{
    POLICE_MISSING_DEPENDENCY("z3");
}

void Z3LP::pop_snapshot()
{
    POLICE_MISSING_DEPENDENCY("z3");
}

LP::model_type Z3LP::get_model() const
{
    POLICE_MISSING_DEPENDENCY("z3");
}

real_t Z3LP::get_objective_value() const
{
    POLICE_MISSING_DEPENDENCY("z3");
}

std::string Z3LP::get_name() const
{
    POLICE_MISSING_DEPENDENCY("z3");
}

real_t Z3LP::get_infinity() const
{
    POLICE_MISSING_DEPENDENCY("z3");
}

void Z3LP::dump(std::ostream&) const
{
    POLICE_MISSING_DEPENDENCY("z3");
}

void Z3LP::dump_model(std::ostream&) const
{
    POLICE_MISSING_DEPENDENCY("z3");
}

} // namespace police

#endif
