#include "police/model.hpp"
#include "police/action.hpp"

#include <algorithm>
#include <sstream>

namespace police {

vector<Action> Model::determinize() const
{
    vector<Action> determinized;
    determinized.reserve(actions.size());
    for (auto i = 0u; i < actions.size(); ++i) {
        const auto& action = actions[i];
        for (const auto& outcome : action.outcomes) {
            determinized.push_back(
                Action(action.label, action.guard, {outcome}));
        }
    }
    return determinized;
}

void Model::report_infos() const
{
    std::cout << "Variables: " << variables.size() << "\n";
    std::cout << "Action labels: " << labels.size() << "\n";
    const auto silent_edges =
        std::count_if(actions.begin(), actions.end(), [](const auto& e) {
            return e.label == SILENT_ACTION;
        });
    std::cout << "Silent actions: " << silent_edges << " ("
              << std::count_if(
                     actions.begin(),
                     actions.begin() + silent_edges,
                     [](const auto& e) { return e.outcomes.size() == 1u; })
              << " deterministic)\n";
    std::cout << "Labeled actions: " << (actions.size() - silent_edges) << " ("
              << std::count_if(
                     actions.begin() + silent_edges,
                     actions.end(),
                     [](const auto& e) { return e.outcomes.size() == 1u; })
              << " deterministic)\n";
}

size_t Model::get_label_id(std::string_view label) const
{
    for (int i = labels.size() - 1; i >= 0; --i) {
        if (labels[i] == label) {
            return i;
        }
    }
    return -1;
}

std::string Model::get_value_name(size_t var, Value value) const
{
    auto value_names = this->value_names.find(var);
    if (value_names == this->value_names.end()) {
        std::ostringstream res;
        res << variables.get_name(var) << "=" << value;
        return res.str();
    } else {
        size_t val_idx = static_cast<int_t>(value);
        assert(val_idx < value_names->second.size());
        return value_names->second[val_idx];
    }
}

} // namespace police
