#pragma once

#include "police/global_arguments.hpp"
#include "police/option_parser.hpp"

#include <string_view>
#include <type_traits>
#include <utility>

namespace police {

struct NoArguments {
    void operator()(const GlobalArguments&, ArgumentsDefinition&) const {}
};

namespace details {
template <typename Definer>
struct NoGlobalsWrapper {
    void operator()(const GlobalArguments&, ArgumentsDefinition& defs) const
    {
        base(defs);
    }

    Definer base;
};

template <typename Definer>
using requires_globals_t = std::integral_constant<
    bool,
    std::is_invocable_v<Definer, const GlobalArguments&, ArgumentsDefinition&>>;

template <typename Definer>
Definer&& make_def_interface(std::true_type, Definer&& definer)
{
    return std::forward<Definer>(definer);
}

template <typename Definer>
NoGlobalsWrapper<std::decay_t<Definer>>
make_def_interface(std::false_type, Definer&& definer)
{
    return {std::forward<Definer>(definer)};
}

template <typename Definer>
auto make_def_interface(Definer&& definer)
{
    return make_def_interface(
        requires_globals_t<Definer>(),
        std::forward<Definer>(definer));
}

} // namespace details

template <typename T>
class Option {
public:
    template <FactoryFunction<T> Factory, typename Definer>
    Option(std::string_view name, Factory&& factory, Definer&& args)
    {
        OptionParser::add_option<T>(
            name,
            std::forward<Factory>(factory),
            details::make_def_interface(std::forward<Definer>(args)));
    }

    template <FactoryFunction<T> Factory>
    Option(std::string_view name, Factory&& factory)
    {
        OptionParser::add_option<T>(
            name,
            std::forward<Factory>(factory),
            NoArguments());
    }
};

template <typename T>
class PointerOption : public Option<std::shared_ptr<T>> {
    using Option<std::shared_ptr<T>>::Option;
};

} // namespace police
