#include "police/option_parser.hpp"
#include "police/argument_store.hpp"
#include "police/global_arguments.hpp"
#include "police/utils/io.hpp"

#include <algorithm>
#include <iterator>
#include <typeindex>

namespace police {

void ArgumentsDefinition::dump(std::ostream& out, int shift)
{
    const std::string ws(shift, ' ');
    for (auto i = 0u; i < keys_.size(); ++i) {
        out << ws << keys_[i] << ": " << arg_types_[i]->name() << "\n";
    }
}

ArgumentsDefinition::EnumArg::EnumArg(
    vector<std::string> values,
    const std::string& default_value)
    : values_(std::move(values))
{
    for (int i = values_.size() - 1; i >= 0; --i) {
        if (values_[i] == default_value) {
            default_ = i;
            break;
        }
    }
}

void ArgumentsDefinition::EnumArg::parse(
    ArgumentStore& store,
    const std::string& key,
    const GlobalArguments&,
    const ParseTree& tree)
{
    for (int i = values_.size() - 1; i >= 0; --i) {
        if (tree.raw == values_[i]) {
            store.set<int>(key, int(i));
            return;
        }
    }
    POLICE_INVALID_ARGUMENT(
        tree.raw,
        "no matching enum entry in " << print_sequence(values_));
}

void ArgumentsDefinition::EnumArg::set_default_value(
    ArgumentStore& store,
    const GlobalArguments&,
    const std::string& key)
{
    store.set<int>(key, int(default_));
}

bool ArgumentsDefinition::EnumArg::has_default_value() const
{
    return default_ >= 0;
}

std::string_view ArgumentsDefinition::EnumArg::name() const
{
    return "enum";
}

void ArgumentsDefinition::EnumArg::validate(
    const GlobalArguments&,
    const ParseTree& tree) const
{
    for (int i = values_.size() - 1; i >= 0; --i) {
        if (tree.raw == values_[i]) {
            return;
        }
    }
    POLICE_INVALID_ARGUMENT(
        tree.raw,
        "no matching enum entry in " << print_sequence(values_));
}

void OptionParser::dump(const GlobalArguments& globals, std::ostream& out)
{
    if (options_.empty()) {
        return;
    }
    vector<std::pair<std::type_index, std::string>> options;
    std::transform(
        options_.begin(),
        options_.end(),
        std::back_inserter(options),
        [](auto&& x) { return x.first; });
    std::sort(options.begin(), options.end());
    for (auto i = 0u; i < options.size(); ++i) {
        if (i == 0u || options[i - 1].first != options[i].first) {
            out << options[i].first.name() << "\n";
        }
        ArgumentsDefinition defs;
        options_.find(options[i])->second.add_arguments(globals, defs);
        out << "  " << options[i].second << "\n";
        defs.dump(out, 4);
        out << "\n";
    }
}

unordered_map<std::pair<std::type_index, std::string>, OptionParser::OptionInfo>
    OptionParser::options_;

} // namespace police
