#include "police/parse_tree.hpp"

#include "police/macros.hpp"

#include <algorithm>
#include <cassert>
#include <cctype>
#include <string_view>

namespace police {

namespace {

const char* skipws(const char* begin, const char* end)
{
    return std::find_if(begin, end, [](char c) { return !std::isspace(c); });
}

std::string_view parse_name(const char*& begin, const char* end)
{
    begin = skipws(begin, end);
    end = std::find_if(begin, end, [](char c) {
        return !std::isalnum(c) && c != '_' && c != '.';
    });
    const auto begin_ = begin;
    begin = end;
    return {begin_, end};
}

std::string_view
parse_string(std::string_view arg, const char*& it, const char* end)
{
    assert(*it == '"');
    ++it;
    const auto begin = it;
    for (; it != end && *it != '"'; it += it != end) {
        it += *it == '\\';
    }
    if (it == end) {
        POLICE_INVALID_ARGUMENT(arg, "missing terminating \"");
    }
    assert(*it == '"');
    ++it;
    return {begin, it - 1};
}

ParseTree parse(std::string_view arg, const char*& begin, const char* end);

void parse_argument(
    ParseTree& parent,
    std::string_view arg,
    const char*& begin,
    const char* end);

void parse_arguments(
    ParseTree& parent,
    std::string_view arg,
    const char*& begin,
    const char* end)
{
    assert(*begin == '(');
    begin = skipws(begin + 1, end);
    for (;;) {
        parse_argument(parent, arg, begin, end);
        begin = skipws(begin, end);
        if (begin == end) {
            POLICE_INVALID_ARGUMENT(arg, "missing closing bracket");
        }
        if (*begin == ')') {
            break;
        }
        if (*begin != ',') {
            POLICE_INVALID_ARGUMENT(
                arg,
                "unexpected character " << *begin
                                        << ". Expected either , or ).");
        }
        begin = skipws(begin + 1, end);
    }
    assert(*begin == ')');
    ++begin;
}

void add_argument(ParseTree& parent, std::string_view arg, ParseTree tree)
{
    if (!parent.kwargs.empty()) {
        POLICE_INVALID_ARGUMENT(
            arg,
            "position arguments must not follow keyword arguments");
    }
    parent.add_argument(std::move(tree));
}

void add_argument(
    ParseTree& parent,
    std::string_view arg,
    std::string_view key,
    ParseTree tree)
{
    if (tree.name.size() == 0u) {
        return;
    }
    if (parent.kwargs.count(std::string(key))) {
        POLICE_INVALID_ARGUMENT(arg, "multiple values for argument " << key);
    }
    parent.add_argument(key, std::move(tree));
}

void parse_argument(
    ParseTree& parent,
    std::string_view arg,
    const char*& begin,
    const char* end)
{
    begin = skipws(begin, end);
    if (begin == end) {
        return;
    }
    if (*begin == '"') {
        add_argument(parent, arg, ParseTree(parse_string(arg, begin, end)));
    } else if (*begin != ')') {
        const auto begin_ = begin;
        auto name = parse_name(begin, end);
        begin = skipws(begin, end);
        if (begin == end || *begin == ',' || *begin == ')') {
            add_argument(parent, arg, ParseTree(name));
        } else if (*begin == '=') {
            ++begin;
            add_argument(parent, arg, name, parse(arg, begin, end));
        } else if (*begin == '(') {
            ParseTree t;
            t.name = name;
            parse_arguments(t, arg, begin, end);
            t.raw = std::string_view(begin_, begin);
            add_argument(parent, arg, std::move(t));
        } else {
            POLICE_INVALID_ARGUMENT(
                arg,
                "unexpected character " << *begin
                                        << ". Expected either = or (.");
        }
    }
}

ParseTree parse(std::string_view arg, const char*& begin, const char* end)
{
    begin = skipws(begin, end);
    if (begin != end && *begin == '"') {
        return ParseTree(parse_string(arg, begin, end));
    }
    const auto begin_ = begin;
    std::string_view name = parse_name(begin, end);
    if (name.empty()) {
        POLICE_INVALID_ARGUMENT(arg, "argument must not be empty");
    }
    begin = skipws(begin, end);
    ParseTree result(name);
    if (begin == end || *begin == ',' || *begin == ')') {
        return result;
    } else if (*begin == '(') {
        parse_arguments(result, arg, begin, end);
        result.raw = std::string_view(begin_, begin);
        return result;
    } else {
        POLICE_INVALID_ARGUMENT(
            arg,
            "unexpected character " << *begin << ". Expected , or (.");
    }
}

} // namespace

ParseTree::ParseTree(std::string_view name)
    : raw(name)
    , name(name)
{
}

void ParseTree::add_argument(ParseTree arg)
{
    args.push_back(std::make_shared<ParseTree>(std::move(arg)));
}

void ParseTree::add_argument(std::string_view key, ParseTree arg)
{
    kwargs.emplace(
        std::string(key),
        std::make_shared<ParseTree>(std::move(arg)));
}

ParseTree ParseTree::parse(std::string_view arg)
{
    auto begin = arg.begin();
    auto result = police::parse(arg, begin, arg.end());
    begin = skipws(begin, arg.end());
    if (begin != arg.end()) {
        POLICE_INVALID_ARGUMENT(
            arg,
            "expected end of argument but got "
                << std::string_view(begin, arg.end()));
    }
    return result;
}

} // namespace police
