#include "police/plan_validation_unit.hpp"

#include "police/addtree_policy.hpp"
#include "police/arguments.hpp"
#include "police/execution_unit.hpp"
#include "police/macros.hpp"
#include "police/model.hpp"
#include "police/nn_policy.hpp"
#include "police/option.hpp"
#include "police/option_parser.hpp"
#include "police/storage/flat_state.hpp"
#include "police/storage/path.hpp"
#include "police/successor_generator/determinize_outcomes_adapator.hpp"
#include "police/successor_generator/successor_generators.hpp"

#include <algorithm>
#include <cctype>
#include <fstream>
#include <ios>
#include <ranges>
#include <string>
#include <string_view>

namespace police {

template <typename SuccessorGenerator>
PlanValidatorUnit<SuccessorGenerator>::PlanValidatorUnit(
    PlanValidator<SuccessorGenerator> validator,
    Path path)
    : validator_(std::move(validator))
    , path_(std::move(path))
{
}

template <typename SuccessorGenerator>
void PlanValidatorUnit<SuccessorGenerator>::run()
{
    const auto result = validator_.validate_state_sequence(
        path_ |
        std::ranges::views::transform([](const auto& pr) { return pr.state; }));
    if (result.status == PlanValidator<SuccessorGenerator>::VALID) {
        std::cout << "Plan is valid." << std::endl;
    } else {
        std::cout << "Plan is invalid." << std::endl;
        std::cout << "validation status: "
                  << validator_.status_to_string(result.status) << " ("
                  << result.status << ")" << std::endl;
        std::cout << "invalid at step: " << result.index << std::endl;
    }
}

namespace {

void skipws(std::string_view::iterator& begin, std::string_view::iterator end)
{
    begin =
        std::find_if_not(begin, end, [](char c) { return std::isspace(c); });
}

std::pair<size_t, Value> parse_variable_value(
    const Model& model,
    std::string_view::iterator& begin,
    std::string_view::iterator end)
{
    auto sep = std::find(begin, end, '=');
    if (sep == end) {
        POLICE_INTERNAL_ERROR(
            "could not parse variable assignment "
            << std::string_view(begin, end));
    }
    auto var_name = std::string_view(begin, sep);
    auto val_end = sep;
    for (; val_end != end && *val_end != 'i' && *val_end != 'd' &&
           *val_end != 'b';
         ++val_end) {
    }
    if (val_end == end) {
        POLICE_INTERNAL_ERROR(
            "could not parse variable assignment "
            << std::string_view(begin, end));
    }
    size_t var = 0;
    for (; var < model.variables.size(); ++var) {
        if (model.variables[var].name == var_name) {
            break;
        }
    }
    if (var == model.variables.size()) {
        POLICE_INTERNAL_ERROR(
            "could not parse variable assignment "
            << std::string_view(begin, end));
    }
    if (*val_end == 'i' || *val_end == 'b') {
        const int val = std::stoi(std::string(sep + 1, val_end));
        begin = val_end + 1;
        return {var, Value(val)};
    } else {
        const real_t val = std::stod(std::string(sep + 1, val_end));
        begin = val_end + 1;
        return {var, Value(val)};
    }
}

flat_state parse_state(const Model& model, std::string_view state_line)
{
    if (state_line.size() < 2) {
        POLICE_INTERNAL_ERROR("could not parse state from empty string");
    }
    if (state_line[0] != '{') {
        POLICE_INTERNAL_ERROR(
            "could not parse state string " << state_line << " expected char "
                                            << "'{'" << " at position 0");
    }
    flat_state state(model.variables.size());
    vector<bool> defined(model.variables.size(), false);
    auto begin = state_line.begin() + 1;
    auto end = state_line.end();
    for (;;) {
        skipws(begin, end);
        auto pr = parse_variable_value(model, begin, end);
        if (defined[pr.first]) {
            POLICE_INTERNAL_ERROR(
                "could not parse state: received multiple values for variable "
                << pr.first << " (" << model.variables[pr.first].name << ")");
        }
        defined[pr.first] = true;
        state[pr.first] = pr.second;
        skipws(begin, end);
        if (begin != end && *begin != ',') {
            break;
        }
        ++begin;
    }
    if (begin == end || *begin != '}') {
        POLICE_INTERNAL_ERROR(
            "could not parse state string " << state_line << " expected char "
                                            << "'}'"
                                            << " but reached end of string");
    }
    ++begin;
    skipws(begin, end);
    if (begin != end) {
        POLICE_INTERNAL_ERROR(
            "could not parse state string " << state_line
                                            << " expected end of string");
    }
    if (!std::all_of(defined.begin(), defined.end(), [](bool b) {
            return b;
        })) {
        POLICE_INTERNAL_ERROR(
            "could not parse state string " << state_line
                                            << " not all variables defined");
    }
    return state;
}

size_t parse_label(const Model& model, std::string_view label_line)
{
    const std::string label(
        label_line.begin(),
        std::find_if(label_line.begin(), label_line.end(), [](char c) {
            return std::isspace(c);
        }));
    for (size_t idx = 0; idx < model.labels.size(); ++idx) {
        if (model.labels[idx] == label) {
            return idx;
        }
    }
    POLICE_INTERNAL_ERROR("label " << label << " not found");
}

Path parse_plan_file(const Model& model, std::string_view file_name)
{
    std::ifstream f;
    f.open(std::string(file_name).c_str());
    if (!f.is_open()) {
        POLICE_INTERNAL_ERROR("could not read file " << file_name);
    }
    Path path;
    std::string line;
    for (;;) {
        if (!std::getline(f, line)) {
            break;
        }
        flat_state state = parse_state(model, line);
        if (!std::getline(f, line)) {
            path.push_back(
                StateLabelPair(std::move(state), StateLabelPair::NO_LABEL));
            break;
        }
        size_t label = parse_label(model, line);
        path.push_back(StateLabelPair(std::move(state), label));
    }
    return path;
}

PointerOption<ExecutionUnit> _opt(
    "validate",
    [](const Arguments& args) -> std::shared_ptr<ExecutionUnit> {
        const Model& model = args.get_model();
        Path path = parse_plan_file(model, args.get<std::string>("plan_file"));
        auto create = [&](auto successor_generator_) {
            auto successor_generator =
                successor_generator::determinize_outcomes_adapator(
                    std::move(successor_generator_));
            return std::make_shared<
                PlanValidatorUnit<decltype(successor_generator)>>(
                PlanValidator<decltype(successor_generator)>(
                    std::move(successor_generator),
                    &args.get_property()),
                std::move(path));
        };
        if (args.has_nn_policy()) {
            const auto& policy = args.get_nn_policy();
            if (args.applicability_masking) {
                return create(
                    successor_generator::MaskedPolicySuccessorGenerator(
                        &model.actions,
                        &policy));
            } else {
                return create(
                    successor_generator::PolicySuccessorGenerator(
                        &model.actions,
                        &policy));
            }
        } else if (args.has_addtree_policy()) {
            const auto& policy = args.get_addtree_policy();
            if (args.applicability_masking) {
                return create(
                    successor_generator::MaskedPolicySuccessorGenerator(
                        &model.actions,
                        &policy));
            } else {
                return create(
                    successor_generator::PolicySuccessorGenerator(
                        &model.actions,
                        &policy));
            }
        } else {
            return create(
                successor_generator::BaseSuccessorGenerator(&model.actions));
        }
    },
    [](ArgumentsDefinition& defs) {
        defs.add_argument<std::string>("plan_file");
    });

} // namespace

} // namespace police
