#pragma once

#include "police/storage/value.hpp"

#include <cassert>
#include <memory>
#include <type_traits>

namespace police {

class SATModel {
public:
    SATModel() = default;

    template <
        typename ConcreteModel,
        typename = std::enable_if_t<!std::is_same_v<ConcreteModel, SATModel>>>
    SATModel(ConcreteModel&& model)
        : model_(
              std::make_shared<ModelInterfaceImpl<std::decay_t<ConcreteModel>>>(
                  std::forward<ConcreteModel>(model)))
    {
    }

    [[nodiscard]]
    Value operator[](size_t var_idx) const
    {
        return get_value(var_idx);
    }

    [[nodiscard]]
    Value get_value(size_t var_idx) const
    {
        assert(var_idx < size());
        return model_->get_value(var_idx);
    }

    [[nodiscard]]
    size_t size() const
    {
        return model_->size();
    }

private:
    struct ModelInterface {
        [[nodiscard]]
        virtual Value get_value(size_t var_idx) = 0;

        [[nodiscard]]
        virtual size_t size() const = 0;
    };

    template <typename ConcreteModel>
    struct ModelInterfaceImpl final : public ModelInterface {
        ModelInterfaceImpl(ConcreteModel model)
            : model(std::move(model))
        {
        }

        [[nodiscard]]
        Value get_value(size_t var_idx)
        {
            return model.get_value(var_idx);
        }

        [[nodiscard]]
        size_t size() const
        {
            return model.size();
        }

        ConcreteModel model;
    };

    std::shared_ptr<ModelInterface> model_ = nullptr;
};

} // namespace police
