#include "police/smt.hpp"

#include <algorithm>

namespace police {

void SMT::push_snapshot()
{
    snapshots_.emplace_back(vars_.size(), constraints_.size());
    do_push_snapshot();
}

void SMT::pop_snapshot()
{
    assert(!snapshots_.empty());
    const auto& snap = snapshots_.back();
    vars_.erase(vars_.begin() + snap.num_vars, vars_.end());
    constraints_.erase(
        constraints_.begin() + snap.num_constraints,
        constraints_.end());
    snapshots_.pop_back();
    do_pop_snapshot();
}

void SMT::clear()
{
    vars_.clear();
    constraints_.clear();
    snapshots_.clear();
    do_clear();
}

size_t SMT::add_variables(const VariableSpace& variables)
{
    assert(variables.size() > 0u);
    const auto result = add_variable(variables[0].name, variables[0].type);
    std::for_each(variables.begin() + 1, variables.end(), [&](const auto& var) {
        add_variable(var.name, var.type);
    });
    return result;
}

size_t SMT::add_variable(std::string_view name, VariableType type)
{
    vars_.add_variable(std::string(name), type);
    return do_add_variable(std::move(name), std::move(type));
}

void SMT::add_constraint(const expressions::Expression& expr)
{
    constraints_.push_back(expr);
    do_add_constraint(expr);
}

size_t SMT::num_variables() const
{
    return vars_.size();
}

void SMT::mirror(SMT& other) const
{
    assert(other.num_variables() == 0u);
    size_t var = 0;
    size_t constraint = 0;
    for (size_t i = 0; i < snapshots_.size(); ++i) {
        const auto& snapshot = snapshots_[i];
        for (; var < snapshot.num_vars; ++var) {
            const auto& v = vars_[var];
            other.add_variable(v.name, v.type);
        }
        for (; constraint < snapshot.num_constraints; ++constraint) {
            other.add_constraint(constraints_[constraint]);
        }
        other.push_snapshot();
    }
    for (; var < vars_.size(); ++var) {
        const auto& v = vars_[var];
        other.add_variable(v.name, v.type);
    }
    for (; constraint < constraints_.size(); ++constraint) {
        other.add_constraint(constraints_[constraint]);
    }
}

} // namespace police
