#include "police/smt_z3.hpp"

#ifdef POLICE_Z3

#include "police/_bits/z3_env.hpp"
#include "police/arguments.hpp"
#include "police/expressions/expression.hpp"
#include "police/option.hpp"
#include "police/smt_factory.hpp"
#include "police/storage/variable_space.hpp"

#include <memory>
#include <string_view>

namespace police {

namespace {
} // namespace

Z3SMT::Z3SMT()
    : solver_(std::make_unique<Z3Solver>())
{
}

void Z3SMT::do_push_snapshot()
{
    solver_->push_snapshot();
}

void Z3SMT::do_pop_snapshot()
{
    solver_->pop_snapshot();
    if (--reset_counter_ == 0) {
        Z3SMT fresh;
        mirror(fresh);
        *this = std::move(fresh);
    }
}

void Z3SMT::do_clear()
{
    solver_.reset(new Z3Solver());
}

size_t Z3SMT::do_add_variable(std::string_view, VariableType type)
{
    solver_->add_variable(type);
    return solver_->get_environment().num_variables() - 1;
}

void Z3SMT::do_add_constraint(const expressions::Expression& expr)
{
    solver_->add_constraint(expr);
}

SMT::Status Z3SMT::solve()
{
    return solver_->check() ? Status::SAT : Status::UNSAT;
}

SMT::Status Z3SMT::solve(const vector<expressions::Expression>& assumptions)
{
    return solver_->check(assumptions) ? Status::SAT : Status::UNSAT;
}

SMT::model_type Z3SMT::get_model() const
{
    return solver_->get_model();
}

vector<expressions::Expression> Z3SMT::get_unsat_core() const
{
    return solver_->get_unsat_core();
}

void Z3SMT::dump(std::ostream& out) const
{
    solver_->dump(out);
}

namespace {
PointerOption<SMTFactory> _option("z3", [](const Arguments&) {
    return std::make_shared<Z3SMTFactory>();
});
} // namespace

} // namespace police

#else

#include "macros.hpp"

namespace police {

Z3SMT::Z3SMT()
{
}

SMT::Status Z3SMT::solve()
{
    POLICE_MISSING_DEPENDENCY("Z3");
}

SMT::Status Z3SMT::solve(const vector<expressions::Expression>&)
{
    POLICE_MISSING_DEPENDENCY("Z3");
}

SMT::model_type Z3SMT::get_model() const
{
    POLICE_MISSING_DEPENDENCY("Z3");
}

vector<expressions::Expression> Z3SMT::get_unsat_core() const
{
    POLICE_MISSING_DEPENDENCY("Z3");
}

void Z3SMT::dump(std::ostream&) const
{
    POLICE_MISSING_DEPENDENCY("Z3");
}

void Z3SMT::do_push_snapshot()
{
    POLICE_MISSING_DEPENDENCY("Z3");
}

void Z3SMT::do_pop_snapshot()
{
    POLICE_MISSING_DEPENDENCY("Z3");
}

void Z3SMT::do_clear()
{
    POLICE_MISSING_DEPENDENCY("Z3");
}

size_t Z3SMT::do_add_variable(std::string_view, VariableType)
{
    POLICE_MISSING_DEPENDENCY("Z3");
}

void Z3SMT::do_add_constraint(const expressions::Expression&)
{
    POLICE_MISSING_DEPENDENCY("Z3");
}

} // namespace police

#endif
