#pragma once

#include "police/storage/flat_state.hpp"
#include "police/storage/state_adapter.hpp"
#include "police/storage/value.hpp"

namespace police {

namespace detail::compressed_state {

template <typename Compressor>
class container {
    using bin_t = typename Compressor::bin_t;

public:
    using int_t = typename Compressor::int_t;
    using float_t = typename Compressor::float_t;
    using size_t = typename Compressor::size_t;

    constexpr container(
        const bin_t* values,
        const Compressor* compressor) noexcept
        : values_(values)
        , compressor_(compressor)
    {
    }

    constexpr void set(size_t index, const Value& value)
    {
        if (get_type(index) == Value::Type::REAL) {
            compressor_->set_float(values_, index, static_cast<float_t>(value));
        } else {
            compressor_->set_int(values_, index, static_cast<int_t>(value));
        }
    }

    constexpr void set_float(size_t index, float_t value) const
    {
        assert(index < compressor_->num_floats());
        compressor_->set_float(values_, index, value);
    }

    constexpr void set_int(size_t index, int_t value) const
    {
        assert(
            index >= compressor_->num_floats() && index < compressor_->size());
        compressor_->set_int(values_, index, value);
    }

    [[nodiscard]]
    constexpr Value get(size_t index) const
    {
        if (get_type(index) == Value::Type::REAL) {
            return Value((real_t)get_float(index));
        } else {
            return Value((int_t)get_int(index));
        }
    }

    [[nodiscard]]
    constexpr float_t get_float(size_t index) const
    {
        assert(index < compressor_->num_floats());
        return compressor_->get_float(values_, index);
    }

    [[nodiscard]]
    constexpr int_t get_int(size_t index) const
    {
        assert(
            index >= compressor_->num_floats() && index < compressor_->size());
        return compressor_->get_int(values_, index);
    }

    [[nodiscard]]
    constexpr Value::Type get_type(size_t index) const
    {
        if (index < compressor_->num_floats()) {
            return Value::Type::REAL;
        }
        return Value::Type::INT;
    }

    [[nodiscard]]
    constexpr size_t size() const
    {
        return compressor_->size();
    }

    [[nodiscard]]
    constexpr const Compressor* compressor() const
    {
        return compressor_;
    }

    [[nodiscard]]
    constexpr const bin_t* data() const
    {
        return values_;
    }

private:
    const bin_t* values_;
    const Compressor* compressor_;
};

} // namespace detail::compressed_state

template <typename Compressor>
using compressed_state =
    state_adapter<detail::compressed_state::container<Compressor>>;

template <typename Compressor>
[[nodiscard]]
constexpr flat_state unpack(const compressed_state<Compressor>& compressed)
{
    auto* compressor = compressed.data()->compressor();
    flat_state flat(compressor->size());
    auto* cdata = compressed.data();
    auto* fdata = flat.data();
    for (auto i = 0u; i < compressor->num_floats(); ++i) {
        fdata->set(i, Value(static_cast<real_t>(cdata->get_float(i))));
    }
    for (auto i = compressor->num_floats(); i < compressed.size(); ++i) {
        fdata->set(i, Value(static_cast<int_t>(cdata->get_int(i))));
    }
    return flat;
}

} // namespace police
