#pragma once

#include "police/utils/bit_operations.hpp"
#include "police/utils/hash.hpp"
#include "police/utils/type_traits.hpp"

#include <algorithm>
#include <cstdint>
#include <iterator>
#include <type_traits>

namespace police {

template <typename Block, typename Allocator>
class dynamic_bit_field;

namespace _bit_field {

using size_type = std::uint32_t;
using difference_type = std::int32_t;

template <typename Block>
struct _utils {
    constexpr static std::uint32_t LOG_WIDTH = bits::get_msb(sizeof(Block) * 8);
    constexpr static std::uint32_t INDEX_PATTERN =
        bits::pattern<std::uint32_t>(LOG_WIDTH);

    [[nodiscard]]
    static constexpr std::uint32_t get_num_blocks(std::uint32_t size) noexcept
    {
        const auto block = get_block_index(size);
        const auto remai = get_local_index(size);
        return block + (remai > 0u);
    }

    [[nodiscard]]
    static constexpr std::uint32_t get_capacity(std::uint32_t size) noexcept
    {
        return get_num_blocks(size) * sizeof(Block) * 8;
    }

    [[nodiscard]]
    static constexpr std::uint32_t get_block_index(std::uint32_t idx) noexcept
    {
        return idx >> (LOG_WIDTH);
    }

    [[nodiscard]]
    static constexpr std::uint32_t get_local_index(std::uint32_t idx) noexcept
    {
        return (idx & INDEX_PATTERN);
    }

    [[nodiscard]]
    static constexpr Block get_bit_pattern(std::uint32_t idx) noexcept
    {
        return static_cast<Block>(1)
               << static_cast<Block>(get_local_index(idx));
    }
};

template <typename Block, bool Const>
class bit_reference {
public:
    using block_pointer = police::make_const_t<Const, Block>*;
    using utils = _utils<Block>;

    constexpr bit_reference(block_pointer bits, std::uint32_t idx) noexcept
        : bits_(bits)
        , idx_(idx)
    {
    }

    template <bool C = !Const, typename = std::enable_if_t<C && C != Const>>
    constexpr operator bit_reference<Block, C>() const noexcept
    {
        return {bits_, idx_};
    }

    constexpr operator bool() const noexcept
    {
        return bits_[utils::get_block_index(idx_)] &
               utils::get_bit_pattern(idx_);
    }

    template <bool C = Const, typename = std::enable_if_t<!C && C == Const>>
    constexpr bit_reference<Block, C>& operator=(bool value) noexcept
    {
        Block& block = bits_[utils::get_block_index(idx_)];
        block = (block & ~utils::get_bit_pattern(idx_)) |
                (static_cast<Block>(value) << utils::get_local_index(idx_));
        return *this;
    }

    [[nodiscard]]
    constexpr bool operator==(bool b) const noexcept
    {
        return static_cast<bool>(*this) == b;
    }

    [[nodiscard]]
    friend constexpr bool operator==(bool b, const bit_reference& ref) noexcept
    {
        return ref == b;
    }

    [[nodiscard]]
    constexpr auto operator<=>(bool b) const noexcept
    {
        return static_cast<bool>(*this) <=> b;
    }

    [[nodiscard]]
    friend constexpr auto operator<=>(bool b, const bit_reference& ref) noexcept
    {
        return static_cast<bool>(ref) <=> b;
    }

private:
    block_pointer bits_;
    std::uint32_t idx_;
};

template <typename Block, bool Const, typename Allocator>
class dynamic_bit_field_reference;

template <typename Block, bool Const>
class dynamic_bit_field_iterator {
public:
    using size_type = _bit_field::size_type;
    using difference_type = _bit_field::difference_type;
    using block_pointer = police::make_const_t<Const, Block>*;
    using value_type = bool;
    using pointer_type = bool*;
    using reference = _bit_field::bit_reference<Block, Const>;
    using const_reference = _bit_field::bit_reference<Block, true>;

    dynamic_bit_field_iterator() noexcept = default;

    dynamic_bit_field_iterator(block_pointer bits, size_type idx) noexcept
        : bits_(bits)
        , idx_(idx)
    {
    }

    template <bool C = !Const, typename = std::enable_if_t<C && C != Const>>
    operator dynamic_bit_field_iterator<Block, C>() const
    {
        return {bits_, idx_};
    }

    [[nodiscard]]
    reference operator*() const
    {
        return bit_reference<Block, Const>(bits_, idx_);
    }

    [[nodiscard]]
    reference operator[](difference_type n) const
    {
        return bit_reference<Block, Const>(bits_, idx_ + n);
    }

    [[nodiscard]]
    bool operator==(const dynamic_bit_field_iterator& other) const
    {
        return idx_ == other.idx_;
    }

    [[nodiscard]]
    auto operator<=>(const dynamic_bit_field_iterator& other) const
    {
        return idx_ <=> other.idx_;
    }

    dynamic_bit_field_iterator& operator++()
    {
        ++idx_;
        return *this;
    }

    dynamic_bit_field_iterator operator++(int)
    {
        dynamic_bit_field_iterator temp(*this);
        ++*this;
        return temp;
    }

    dynamic_bit_field_iterator& operator--()
    {
        --idx_;
        return *this;
    }

    dynamic_bit_field_iterator operator--(int)
    {
        dynamic_bit_field_iterator temp(*this);
        --*this;
        return temp;
    }

    dynamic_bit_field_iterator& operator+=(difference_type n)
    {
        idx_ += n;
        return *this;
    }

    dynamic_bit_field_iterator& operator-=(difference_type n)
    {
        idx_ -= n;
        return *this;
    }

    [[nodiscard]]
    dynamic_bit_field_iterator operator+(difference_type n) const
    {
        dynamic_bit_field_iterator temp(*this);
        temp += n;
        return temp;
    }

    [[nodiscard]]
    dynamic_bit_field_iterator operator-(difference_type n) const
    {
        dynamic_bit_field_iterator temp(*this);
        temp -= n;
        return temp;
    }

    [[nodiscard]]
    difference_type operator-(const dynamic_bit_field_iterator& it) const
    {
        return static_cast<difference_type>(idx_) -
               static_cast<difference_type>(it.idx_);
    }

    [[nodiscard]]
    friend dynamic_bit_field_iterator
    operator+(difference_type n, const dynamic_bit_field_iterator& it)
    {
        return it + n;
    }

    [[nodiscard]]
    size_type index() const
    {
        return idx_;
    }

private:
    template <typename, bool, typename>
    friend class dynamic_bit_field_reference;

    template <typename, typename>
    friend class police::dynamic_bit_field;

    block_pointer bits_{};
    size_type idx_{};
};

template <typename Block, bool Const>
class active_bit_iterator {
public:
    using size_type = _bit_field::size_type;
    using difference_type = _bit_field::difference_type;
    using block_pointer = police::make_const_t<Const, Block>*;
    using value_type = bool;
    using pointer_type = bool*;
    using reference = bit_reference<Block, Const>;
    using const_reference = bit_reference<Block, true>;
    using utils = _bit_field::_utils<Block>;

    active_bit_iterator() noexcept = default;

    active_bit_iterator(
        block_pointer bits,
        size_type idx,
        size_type size) noexcept
        : bits_(bits)
        , idx_(idx)
        , size_(size)
    {
    }

    operator dynamic_bit_field_iterator<Block, Const>() const
    {
        return {bits_, idx_, size_};
    }

    template <bool C = !Const, typename = std::enable_if_t<C && C != Const>>
    operator active_bit_iterator<Block, C>() const
    {
        return {bits_, idx_, size_};
    }

    [[nodiscard]]
    reference operator*() const
    {
        return bit_reference<Block, Const>(bits_, idx_);
    }

    [[nodiscard]]
    bool operator==(const active_bit_iterator& other) const
    {
        return idx_ == other.idx_;
    }

    [[nodiscard]]
    auto operator<=>(const active_bit_iterator& other) const
    {
        return idx_ <=> other.idx_;
    }

    template <bool C>
    [[nodiscard]]
    bool operator==(const dynamic_bit_field_iterator<Block, C>& it) const
    {
        return idx_ == it.index();
    }

    template <bool C>
    [[nodiscard]]
    auto operator<=>(const dynamic_bit_field_iterator<Block, C>& other) const
    {
        return idx_ <=> other.index();
    }

    active_bit_iterator& operator++()
    {
        ++idx_;
        auto bi = utils::get_block_index(idx_);
        auto li = utils::get_local_index(idx_);
        while (idx_ < size_) {
            const auto b = bits_[bi] >> li;
            if (!b) {
                idx_ = std::min(
                    size_,
                    static_cast<size_type>(sizeof(Block) * 8) * (bi + 1));
                li = 0;
                ++bi;
            } else {
                idx_ += bits::get_lsb(b);
                break;
            }
        }
        return *this;
    }

    active_bit_iterator operator++(int)
    {
        active_bit_iterator temp(*this);
        ++*this;
        return temp;
    }

    [[nodiscard]]
    size_type index() const
    {
        return idx_;
    }

private:
    template <typename, bool, typename>
    friend class dynamic_bit_field_reference;

    template <typename, typename>
    friend class police::dynamic_bit_field;

    block_pointer bits_{};
    size_type idx_{};
    size_type size_{};
};

static_assert(std::input_or_output_iterator<
              dynamic_bit_field_iterator<std::uint64_t, true>>);

template <typename Block, bool Const, typename Allocator>
class dynamic_bit_field_reference {
public:
    using block_pointer = police::make_const_t<Const, Block>*;
    using size_type = std::uint32_t;
    using iterator = _bit_field::dynamic_bit_field_iterator<Block, Const>;
    using const_iterator = _bit_field::dynamic_bit_field_iterator<Block, true>;
    using active_bit_iterator = _bit_field::active_bit_iterator<Block, Const>;
    using const_active_bit_iterator =
        _bit_field::active_bit_iterator<Block, true>;
    using value_type = bool;
    using reference = _bit_field::bit_reference<Block, Const>;
    using const_reference = _bit_field::bit_reference<Block, true>;
    using utils = _bit_field::_utils<Block>;
    using allocator_type = Allocator;

    dynamic_bit_field_reference(
        block_pointer bits,
        size_type capacity,
        size_type size,
        Allocator allocate)
        : bits_(bits)
        , capacity_(capacity)
        , size_(size)
        , allocator_(std::move(allocate))
    {
    }

    [[nodiscard]]
    size_type size() const
    {
        return size_;
    }

    [[nodiscard]]
    size_type capacity() const
    {
        return capacity_;
    }

    [[nodiscard]]
    bool empty() const
    {
        return size_ == 0u;
    }

    [[nodiscard]]
    iterator begin()
    {
        return {bits_, 0u};
    }

    [[nodiscard]]
    iterator end()
    {
        return {bits_, size_};
    }

    [[nodiscard]]
    const_iterator begin() const
    {
        return {bits_, 0u};
    }

    [[nodiscard]]
    const_iterator end() const
    {
        return {bits_, size_};
    }

    [[nodiscard]]
    const_iterator cbegin() const
    {
        return {bits_, 0u};
    }

    [[nodiscard]]
    const_iterator cend() const
    {
        return {bits_, size_};
    }

    [[nodiscard]]
    active_bit_iterator first1()
    {
        active_bit_iterator it(bits_, -1, size_);
        ++it;
        return it;
    }

    [[nodiscard]]
    reference back()
    {
        return _bit_field::bit_reference<Block, Const>(bits_, size_ - 1);
    }

    [[nodiscard]]
    reference front()
    {
        return _bit_field::bit_reference<Block, Const>(bits_, 0);
    }

    [[nodiscard]]
    const_reference back() const
    {
        return _bit_field::bit_reference<Block, true>(bits_, size_ - 1);
    }

    [[nodiscard]]
    const_reference front() const
    {
        return _bit_field::bit_reference<Block, true>(bits_, 0);
    }

    [[nodiscard]]
    reference operator[](size_type pos)
    {
        return _bit_field::bit_reference<Block, Const>(bits_, pos);
    }

    [[nodiscard]]
    const_reference operator[](size_type pos) const
    {
        return _bit_field::bit_reference<Block, true>(bits_, pos);
    }

    [[nodiscard]]
    reference at(size_type pos)
    {
        return _bit_field::bit_reference<Block, Const>(bits_, pos);
    }

    [[nodiscard]]
    const_reference at(size_type pos) const
    {
        return _bit_field::bit_reference<Block, true>(bits_, pos);
    }

    [[nodiscard]]
    bool operator==(const dynamic_bit_field_reference& other) const
    {
        if (size_ != other.size_) {
            return false;
        }
        if (bits_ == other.bits_ || size_ == 0) {
            return true;
        }
        const auto b = utils::get_block_index(size_ - 1);
        const auto i = utils::get_local_index(size_ - 1);
        const auto p = bits::pattern<Block>(i + 1);
        return (bits_[b] & p) == (other.bits_[b] & p) &&
               std::equal(
                   bits_,
                   bits_ + b - 1,
                   other.bits_,
                   other.bits_ + b - 1);
    }

protected:
    template <typename T>
    friend struct police::hash;

    block_pointer bits_;
    size_type capacity_;
    size_type size_;
    Allocator allocator_;
};

} // namespace _bit_field

template <
    typename Block = std::uint64_t,
    typename Allocator = std::allocator<Block>>
using dynamic_bit_field_reference =
    _bit_field::dynamic_bit_field_reference<Block, true, Allocator>;

template <
    typename Block = std::uint64_t,
    typename Allocator = std::allocator<Block>>
class dynamic_bit_field
    : public _bit_field::dynamic_bit_field_reference<Block, false, Allocator> {
    using ref =
        _bit_field::dynamic_bit_field_reference<Block, false, Allocator>;
    using utils = _bit_field::_utils<Block>;

public:
    using const_iterator = ref::const_iterator;
    using iterator = ref::iterator;
    using active_bit_iterator = ref::active_bit_iterator;
    using const_active_bit_iterator = ref::const_active_bit_iterator;
    using reference = ref::reference;
    using const_reference = ref::const_reference;
    using size_type = ref::size_type;
    using value_type = ref::value_type;

    explicit dynamic_bit_field(
        size_type size = 0,
        bool value = {},
        Allocator allocator = {})
        : ref(nullptr, 0, 0, std::move(allocator))
    {
        resize(size, value);
    }

    dynamic_bit_field(const dynamic_bit_field& other)
        : ref(nullptr, 0, 0, other.allocator_)
    {
        resize(other.size(), false);
        std::copy(
            other.bits_,
            other.bits_ + utils::get_num_blocks(other.size()),
            this->bits_);
    }

    dynamic_bit_field(dynamic_bit_field&& other)
        : ref(other.bits_, other.size_, other.capacity_, other.allocator_)
    {
        other.capacity_ = 0;
        other.size_ = 0;
    }

    ~dynamic_bit_field()
    {
        if (this->capacity_) {
            this->allocator_.deallocate(
                this->bits_,
                utils::get_num_blocks(this->capacity_));
        }
    }

    void reserve(size_type capacity)
    {
        if (capacity > this->capacity_) {
            const size_type old_num_blocks =
                utils::get_num_blocks(this->capacity_);
            const size_type new_num_blocks = utils::get_num_blocks(capacity);
            Block* new_bits = this->allocator_.allocate(new_num_blocks);
            if (this->capacity_) {
                std::copy(this->bits_, this->bits_ + old_num_blocks, new_bits);
                this->allocator_.deallocate(this->bits_, old_num_blocks);
            }
            this->bits_ = new_bits;
            this->capacity_ = new_num_blocks * sizeof(Block) * 8;
        }
    }

    void resize(size_type new_size, bool value = {})
    {
        if (new_size > this->size_) {
            reserve(new_size);
            assign({this->bits_, this->size_}, {this->bits_, new_size}, value);
        }
        this->size_ = new_size;
    }

    void clear() { this->size_ = 0; }

    void assign(iterator first, iterator last, bool value)
    {
        if (value) {
            set_true(first, last);
        } else {
            set_false(first, last);
        }
    }

    void set_false(iterator first, iterator last)
    {
        if (first == last) {
            return;
        }
        const auto idx = utils::get_local_index(first.idx_);
        const auto block = utils::get_block_index(first.idx_);
        const auto new_block = utils::get_block_index(last.idx_ - 1);
        const auto new_idx = utils::get_local_index(last.idx_ - 1);
        if (block == new_block) {
            this->bits_[block] = this->bits_[block] &
                                 ~bits::pattern<Block>(new_idx - idx + 1, idx);
        } else {
            this->bits_[block] =
                this->bits_[block] &
                ~bits::pattern<Block>(sizeof(Block) * 8 - idx, idx);
            std::fill(
                this->bits_ + block + 1,
                this->bits_ + new_block,
                static_cast<Block>(0));
            this->bits_[new_block] =
                this->bits_[new_block] & ~bits::pattern<Block>(new_idx + 1);
        }
    }

    void set_true(iterator first, iterator last)
    {
        if (first == last) {
            return;
        }
        const auto idx = utils::get_local_index(first.idx_);
        const auto block = utils::get_block_index(first.idx_);
        const auto new_block = utils::get_block_index(last.idx_ - 1);
        const auto new_idx = utils::get_local_index(last.idx_ - 1);
        if (block == new_block) {
            this->bits_[block] = this->bits_[block] |
                                 bits::pattern<Block>(new_idx - idx + 1, idx);
        } else {
            this->bits_[block] =
                this->bits_[block] |
                bits::pattern<Block>(sizeof(Block) * 8 - idx, idx);
            std::fill(
                this->bits_ + block + 1,
                this->bits_ + new_block,
                ~static_cast<Block>(0));
            this->bits_[new_block] =
                this->bits_[new_block] | bits::pattern<Block>(new_idx + 1);
        }
    }

    void push_back(bool value)
    {
        if (this->size_ == this->capacity_) {
            reserve((this->size_ + 1u) * 1.33);
        }
        _bit_field::bit_reference<Block, false>(this->bits_, this->size_) =
            value;
        ++this->size_;
    }

    void pop_back() { --this->size_; }
};

template <typename Block, bool Const, typename Allocator>
struct hash<_bit_field::dynamic_bit_field_reference<Block, Const, Allocator>> {
    [[nodiscard]]
    std::size_t operator()(
        const _bit_field::dynamic_bit_field_reference<Block, Const, Allocator>&
            bits) const
    {
        if (bits.empty()) {
            return 0;
        }
        police::hash<Block> block_hash;
        int b = _bit_field::_utils<Block>::get_block_index(bits.size() - 1);
        const auto p = bits::pattern<Block>(
            _bit_field::_utils<Block>::get_local_index(bits.size() - 1));
        std::size_t h = block_hash(bits.bits_[b] & p);
        --b;
        for (; b >= 0; --b) {
            h = hash_combine(h, block_hash(bits.bits_[b]));
        }
        return h;
    }
};

template <typename Block, typename Allocator>
struct hash<dynamic_bit_field_reference<Block, Allocator>> {

    [[nodiscard]]
    std::size_t
    operator()(const dynamic_bit_field<Block, Allocator>& field) const
    {
        return hash<
            _bit_field::dynamic_bit_field_reference<Block, false, Allocator>>(
            field);
    }
};

} // namespace police
