#include "police/storage/matrix.hpp"

#ifndef POLICE_EIGEN

#include "police/storage/lin_vector.hpp"

namespace police {

Matrix::Matrix()
    : cols_(0)
    , rows_(0)
{
}

Matrix::Matrix(size_t rows, size_t cols)
    : data_(rows * cols)
    , cols_(cols)
    , rows_(rows)
{
}

void Matrix::setZero()
{
    std::fill(data_.begin(), data_.end(), 0);
}

LinVector Matrix::operator*(const LinVector& vec) const
{
    LinVector res(rows_);
    for (int r = rows() - 1; r >= 0; --r) {
        real_t val = 0;
        const real_t* coef = &data_[r * cols() + cols() - 1];
        for (int c = cols() - 1; c >= 0; --c, --coef) {
            val += *coef * vec[c];
        }
        res[r] = val;
    }
    return res;
}

Matrix Matrix::operator+(const Matrix& mat) const
{
    Matrix result(rows(), cols());
    const real_t* x = &data_[rows() * cols() - 1];
    const real_t* y = &mat.data_[rows() * cols() - 1];
    for (int r = rows() - 1; r >= 0; --r) {
        for (int c = cols() - 1; c >= 0; --c, --x, --y) {
            result(r, c) = *x + *y;
        }
    }
    return result;
}

Matrix Matrix::operator-(const Matrix& mat) const
{
    Matrix result(rows(), cols());
    const real_t* x = &data_[rows() * cols() - 1];
    const real_t* y = &mat.data_[rows() * cols() - 1];
    for (int r = rows() - 1; r >= 0; --r) {
        for (int c = cols() - 1; c >= 0; --c, --x, --y) {
            result(r, c) = *x - *y;
        }
    }
    return result;
}

Matrix Matrix::operator*(const Matrix& mat) const
{
    Matrix result(rows(), mat.cols());
    for (int r = rows() - 1; r >= 0; --r) {
        for (int c = mat.cols() - 1; c >= 0; --c) {
            real_t res = 0;
            const real_t* x = &data_[r * cols() + cols() - 1];
            for (int i = cols() - 1; i >= 0; --i, --x) {
                res += *x * mat(i, c);
            }
            result(r, c) = res;
        }
    }
    return result;
}

Matrix Matrix::operator*(real_t scalar) const
{
    Matrix result(rows(), cols());
    const real_t* x = &data_[rows() * cols() - 1];
    for (int r = rows() - 1; r >= 0; --r) {
        for (int c = cols() - 1; c >= 0; --c, --x) {
            result(r, c) = *x * scalar;
        }
    }
    return result;
}

Matrix Matrix::operator/(real_t scalar) const
{
    return operator*(1. / scalar);
}

Matrix Matrix::operator-() const
{
    return operator*(-1.);
}

Matrix operator*(real_t scalar, const Matrix& matrix)
{
    return matrix * scalar;
}

} // namespace police

#endif
