#pragma once

#include "police/base_types.hpp"
#include "police/macros.hpp"
#include "police/storage/value.hpp"

#include <algorithm>
#include <concepts>
#include <cstdint>
#include <iterator>

namespace police {

template <typename T>
concept ValueContainer = requires(T t, size_t idx) {
    { t.get(idx) } -> std::convertible_to<Value>;
    { t.size() } -> std::convertible_to<police::size_t>;
};

template <typename T>
concept WritableValueContainer =
    ValueContainer<T> &&
    requires(T t, size_t idx, Value val) { t.set(idx, val); };

namespace detail::state_view {

template <WritableValueContainer ValueContainer>
struct reference : public Value {
public:
    constexpr reference(ValueContainer* container, std::size_t index) noexcept
        : Value(container->get(index))
        , container_(container)
        , index_(index)
    {
    }

    constexpr reference& operator=(const Value& value)
    {
        Value::operator=(value);
        container_->set(index_, value);
        return *this;
    }

private:
    ValueContainer* container_;
    std::size_t index_;
};

template <typename C>
constexpr bool operator==(const reference<C>& ref, const Value& value)
{
    return static_cast<Value>(ref) == value;
}

template <typename C>
constexpr bool operator==(const Value& value, const reference<C>& ref)
{
    return static_cast<Value>(ref) == value;
}

template <typename C1, typename C2>
constexpr bool operator==(const reference<C1>& ref1, const reference<C2>& ref2)
{
    return static_cast<Value>(ref1) == static_cast<Value>(ref2);
}

template <typename ValueContainer, typename Reference>
struct iterator {
public:
    using value_type = Value;
    using difference_type = std::int32_t;

    constexpr iterator() noexcept = default;

    constexpr iterator(ValueContainer* container, police::size_t index) noexcept
        : container_(container)
        , index_(index)
    {
    }

    [[nodiscard]]
    constexpr auto operator*() const
    {
        return Reference()(container_, index_);
    }

    [[nodiscard]]
    constexpr auto operator[](difference_type x) const
    {
        return *(*this + x);
    }

    [[nodiscard]]
    constexpr bool operator==(const iterator& other) const
    {
        return index_ == other.index_;
    }

    [[nodiscard]]
    constexpr auto operator<=>(const iterator& other) const
    {
        return index_ <=> other.index_;
    }

    constexpr iterator& operator++()
    {
        ++index_;
        return *this;
    }

    constexpr iterator operator++(int)
    {
        iterator copy(*this);
        ++*this;
        return copy;
    }

    constexpr iterator& operator--()
    {
        --index_;
        return *this;
    }

    constexpr iterator operator--(int)
    {
        iterator copy(*this);
        --*this;
        return copy;
    }

    constexpr iterator& operator+=(difference_type x)
    {
        index_ += x;
        return *this;
    }

    [[nodiscard]]
    constexpr iterator operator+(difference_type x) const
    {
        iterator copy(*this);
        copy += x;
        return copy;
    }

    constexpr iterator& operator-=(difference_type x)
    {
        index_ -= x;
        return *this;
    }

    [[nodiscard]]
    constexpr iterator operator-(difference_type x) const
    {
        iterator copy(*this);
        copy -= x;
        return copy;
    }

    [[nodiscard]]
    constexpr difference_type operator-(const iterator& other) const
    {
        return static_cast<difference_type>(index_) -
               static_cast<difference_type>(other.index_);
    }

    [[nodiscard]]
    friend iterator operator+(difference_type n, const iterator& iter)
    {
        return iter + n;
    }

    [[nodiscard]]
    friend iterator operator-(difference_type n, const iterator& iter)
    {
        return iter - n;
    }

private:
    ValueContainer* container_ = nullptr;
    std::uint32_t index_ = 0;
};

template <WritableValueContainer Container>
struct ReferenceFactory {
    [[nodiscard]]
    constexpr reference<Container>
    operator()(Container* container, std::uint32_t index) const
    {
        return reference(container, index);
    }
};

struct ValueFactory {
    template <ValueContainer Container>

    [[nodiscard]]
    constexpr Value operator()(Container* container, std::uint32_t index) const
    {
        return container->get(index);
    }
};

template <typename Container>
using ref_iterator = iterator<Container, ReferenceFactory<Container>>;

template <ValueContainer Container>
using value_iterator = iterator<const Container, ValueFactory>;

} // namespace detail::state_view

template <ValueContainer Container>
struct state_adapter : private Container {
    constexpr static bool IsConst = !WritableValueContainer<Container>;

public:
    using value = Value;
    using reference =
        police::ite_t<IsConst, Value, detail::state_view::reference<Container>>;
    using const_iterator = detail::state_view::value_iterator<Container>;
    using iterator = police::ite_t<
        IsConst,
        const_iterator,
        detail::state_view::ref_iterator<Container>>;
    static_assert(std::bidirectional_iterator<const_iterator>);
    static_assert(std::bidirectional_iterator<iterator>);

    using Container::Container;

    using Container::size;

    [[nodiscard]]
    constexpr value operator[](size_t index) const
    {
        return get(index);
    }

    [[nodiscard]]
    constexpr reference operator[](size_t index)
    {
        if constexpr (IsConst) {
            return get(index);
        } else {
            return {this, index};
        }
    }

    [[nodiscard]]
    constexpr iterator begin()
    {
        return {this, 0};
    }

    [[nodiscard]]
    constexpr iterator end()
    {
        return {this, size()};
    }

    [[nodiscard]]
    constexpr const_iterator begin() const
    {
        return {this, 0};
    }

    [[nodiscard]]
    constexpr const_iterator end() const
    {
        return {this, size()};
    }

    [[nodiscard]]
    constexpr const_iterator cbegin() const
    {
        return {this, 0};
    }

    [[nodiscard]]
    constexpr const_iterator cend() const
    {
        return {this, size()};
    }

    [[nodiscard]]
    constexpr Container* data()
    {
        return this;
    }

    [[nodiscard]]
    constexpr const Container* data() const
    {
        return this;
    }

    template <typename C>
    [[nodiscard]]
    constexpr bool operator==(const state_adapter<C>& other) const
    {
        if (size() != other.size()) {
            return false;
        }
        return std::ranges::equal(*this, other);
    }

private:
    constexpr Value get(size_t index) const
    {
        return detail::state_view::ValueFactory()(data(), index);
    }
};

template <typename Container>
std::ostream& operator<<(
    std::ostream& out,
    const detail::state_view::reference<Container>& ref)
{
    return out << static_cast<Value>(ref);
}

} // namespace police
