#include "police/storage/state_registry.hpp"
#include "police/storage/variable_space.hpp"

namespace police {

namespace {

struct IsReal {
    template <typename T>
    bool operator()(T&& var) const
    {
        return var.type.is_real() || var.type.is_bounded_real();
    }
};

size_t get_num_floats(const VariableSpace& variables)
{
    assert(std::is_partitioned(variables.begin(), variables.end(), IsReal()));
    auto it =
        std::partition_point(variables.begin(), variables.end(), IsReal());
    return std::distance(variables.begin(), it);
}

vector<BoundedIntType> get_integers(const VariableSpace& variables)
{
    assert(std::is_partitioned(variables.begin(), variables.end(), IsReal()));
    auto it =
        std::partition_point(variables.begin(), variables.end(), IsReal());
    vector<BoundedIntType> bounds;
    for (; it != variables.end(); ++it) {
        std::visit(
            [&](auto&& t) {
                using T = std::decay_t<decltype(t)>;
                if constexpr (std::is_same_v<T, BoolType>) {
                    bounds.emplace_back(0, 1);
                } else if constexpr (std::is_same_v<T, BoundedIntType>) {
                    bounds.emplace_back(t.lower_bound, t.upper_bound);
                } else {
                    bounds.emplace_back();
                }
            },
            (*it).type);
    }
    return bounds;
}

} // namespace

StateRegistry::StateRegistry(const VariableSpace& variables)
    : StateRegistry(get_num_floats(variables), get_integers(variables))
{
}

StateRegistry::StateRegistry(
    size_t floats,
    const vector<BoundedIntType>& integers)
    : explicit_state_container<>(floats, integers.begin(), integers.end())
{
}

} // namespace police
