#include "police/storage/variable_space.hpp"

#include <iostream>
#include <type_traits>

namespace police {

size_t VariableSpace::size() const
{
    return types_.size();
}

size_t VariableSpace::add_variable(identifier_name_t name, VariableType type)
{
    types_.push_back(std::move(type));
    names_.push_back(std::move(name));
    return size() - 1;
}

VariableSpace::reference VariableSpace::operator[](size_t var_id)
{
    return {var_id, types_[var_id], names_[var_id]};
}

VariableSpace::const_reference VariableSpace::operator[](size_t var_id) const
{
    return {var_id, types_[var_id], names_[var_id]};
}

identifier_name_t& VariableSpace::get_name(size_t var_id)
{
    return names_[var_id];
}

VariableType& VariableSpace::get_type(size_t var_id)
{
    return types_[var_id];
}

const identifier_name_t& VariableSpace::get_name(size_t var_id) const
{
    return names_[var_id];
}

const VariableType& VariableSpace::get_type(size_t var_id) const
{
    return types_[var_id];
}

VariableSpace::iterator VariableSpace::begin()
{
    return iterator(this, 0);
}

VariableSpace::iterator VariableSpace::end()
{
    return iterator(this, size());
}

VariableSpace::const_iterator VariableSpace::begin() const
{
    return const_iterator(this, 0);
}

VariableSpace::const_iterator VariableSpace::end() const
{
    return const_iterator(this, size());
}

VariableSpace::const_iterator VariableSpace::cbegin() const
{
    return begin();
}

VariableSpace::const_iterator VariableSpace::cend() const
{
    return end();
}

void VariableSpace::erase(iterator pos)
{
    const size_t idx = pos.idx_;
    types_.erase(types_.begin() + idx);
    names_.erase(names_.begin() + idx);
}

void VariableSpace::erase(iterator begin, iterator end)
{
    const size_t i = begin.idx_;
    const size_t j = end.idx_;
    types_.erase(types_.begin() + i, types_.begin() + j);
    names_.erase(names_.begin() + i, names_.begin() + j);
}

VariableSpace::iterator
VariableSpace::insert(const_iterator begin, const_iterator end)
{
    const auto idx = size();
    for (; begin != end; ++begin) {
        types_.push_back(begin->type);
        names_.push_back(begin->name);
    }
    return iterator(this, idx);
}

void VariableSpace::clear()
{
    types_.clear();
    names_.clear();
}

size_t VariableSpace::get_variable_id(std::string_view name) const
{
    for (size_t i = 0; i < names_.size(); ++i) {
        if (names_[i] == name) {
            return i;
        }
    }
    return -1;
}

std::ostream& operator<<(std::ostream& out, const police::VariableType& type)
{
    std::visit(
        [&](auto&& t) {
            using T = std::decay_t<decltype(t)>;
            if constexpr (std::is_same_v<T, police::BoolType>) {
                out << "Bool";
            } else if constexpr (std::is_same_v<T, police::IntegerType>) {
                out << "Integer";
            } else if constexpr (std::is_same_v<T, police::RealType>) {
                out << "Real";
            } else if constexpr (std::is_same_v<T, police::BoundedIntType>) {
                out << "BoundedInt(" << t.lower_bound << ", " << t.upper_bound
                    << ")";
            } else if constexpr (std::is_same_v<T, police::BoundedRealType>) {
                out << "BoundedReal(" << t.lower_bound << ", " << t.upper_bound
                    << ")";
            }
        },
        type);
    return out;
}

} // namespace police
