#pragma once

#include "police/storage/state_registry.hpp"

#include <ranges>
#include <utility>

namespace police::successor_generator {

struct CompressedSuccessor {
    CompressedState state;
    size_t label;
};

template <typename SuccessorIterator>
class CompressedSuccessorIterator {
public:
    using difference_type = int;
    using value_type = CompressedSuccessor;

    explicit CompressedSuccessorIterator(
        SuccessorIterator it,
        StateRegistry* state_registry)
        : it_(std::move(it))
        , state_registry_(state_registry)
    {
    }

    CompressedSuccessorIterator() = default;

    [[nodiscard]]
    CompressedSuccessor operator*() const
    {
        auto succ = *it_;
        auto ref = state_registry_->insert(succ.state);
        return CompressedSuccessor(
            CompressedState(ref.first, (*state_registry_)[ref.first]),
            succ.label

        );
    }

    CompressedSuccessorIterator& operator++()
    {
        ++it_;
        return *this;
    }

    CompressedSuccessorIterator operator++(int)
    {
        CompressedSuccessorIterator temp(*this);
        ++it_;
        return temp;
    }

    [[nodiscard]]
    bool operator==(const CompressedSuccessorIterator& other) const
    {
        return it_ == other.it_;
    }

private:
    SuccessorIterator it_{};
    StateRegistry* state_registry_ = nullptr;
};

template <typename SuccessorRange>
class CompressedSuccessorRange
    : public std::ranges::view_interface<
          CompressedSuccessorRange<SuccessorRange>> {
    using internal_iterator = decltype(std::declval<SuccessorRange>().begin());

public:
    using iterator = CompressedSuccessorIterator<internal_iterator>;

    CompressedSuccessorRange() = default;

    CompressedSuccessorRange(
        SuccessorRange range,
        StateRegistry* state_registry)
        : begin_(range.begin(), state_registry)
        , end_(range.end(), state_registry)
    {
    }

    [[nodiscard]]
    iterator begin() const
    {
        return begin_;
    }

    [[nodiscard]]
    iterator end() const
    {
        return end_;
    }

private:
    iterator begin_{};
    iterator end_{};
};

template <typename Iterator>
class CompressedOutcomesIterator {
public:
    using difference_type = int;
    using sub_value_type = decltype(*std::declval<Iterator>());
    using value_type = CompressedSuccessorRange<sub_value_type>;

    CompressedOutcomesIterator() = default;

    CompressedOutcomesIterator(StateRegistry* state_registry, Iterator it)
        : state_registry_(state_registry)
        , it_(std::move(it))
    {
    }

    [[nodiscard]]
    value_type operator*() const
    {
        return CompressedSuccessorRange(*it_, state_registry_);
    }

    CompressedOutcomesIterator& operator++()
    {
        ++it_;
        return *this;
    }

    CompressedOutcomesIterator operator++(int)
    {
        CompressedOutcomesIterator temp(*this);
        ++it_;
        return temp;
    }

    [[nodiscard]]
    bool operator==(const CompressedOutcomesIterator& other) const
    {
        return it_ == other.it_;
    }

private:
    StateRegistry* state_registry_ = nullptr;
    Iterator it_{};
};

template <typename Iterator>
class CompressedOutcomesRange
    : public std::ranges::view_interface<CompressedOutcomesRange<Iterator>> {
public:
    CompressedOutcomesRange() = default;

    CompressedOutcomesRange(
        CompressedOutcomesIterator<Iterator> first,
        CompressedOutcomesIterator<Iterator> last)
        : first_(std::move(first))
        , last_(std::move(last))
    {
    }

    [[nodiscard]]
    CompressedOutcomesIterator<Iterator> begin() const
    {
        return first_;
    }

    [[nodiscard]]
    CompressedOutcomesIterator<Iterator> end() const
    {
        return last_;
    }

private:
    CompressedOutcomesIterator<Iterator> first_;
    CompressedOutcomesIterator<Iterator> last_;
};

template <typename SuccessorGenerator>
class compress_state_adaptor : public SuccessorGenerator {
public:
    compress_state_adaptor(
        SuccessorGenerator successor_generator,
        StateRegistry* state_registry)
        : SuccessorGenerator(std::move(successor_generator))
        , state_registry_(state_registry)
    {
    }

    template <typename State>
    [[nodiscard]]
    auto operator()(const State& state)
    {
        auto outcomes = SuccessorGenerator::operator()(state);
        return CompressedOutcomesRange(
            CompressedOutcomesIterator(state_registry_, outcomes.begin()),
            CompressedOutcomesIterator(state_registry_, outcomes.end()));
    }

private:
    StateRegistry* state_registry_;
};

} // namespace police::successor_generator
