#pragma once

#include "police/action.hpp"
#include "police/storage/state_registry.hpp"
#include "police/successor_generator/applicable_actions_generator.hpp"

#include <iterator>
#include <ranges>

namespace police::successor_generator {

struct Successor {
    FlatState state;
    size_t label;
};

class SuccessorIterator {
public:
    using difference_type = int;
    using value_type = Successor;

    SuccessorIterator() = default;

    [[nodiscard]]
    Successor operator*() const;

    SuccessorIterator& operator++()
    {
        ++outcome_;
        return *this;
    }

    SuccessorIterator operator++(int)
    {
        SuccessorIterator temp(*this);
        ++outcome_;
        return temp;
    }

    [[nodiscard]]
    bool operator==(const SuccessorIterator& other) const
    {
        return outcome_ == other.outcome_;
    }

private:
    friend class OutcomeSuccessorGenerator;
    friend class ApplicableActionsIterator;

    SuccessorIterator(
        vector<Outcome>::const_iterator outcome,
        std::shared_ptr<FlatState> state,
        size_t label)
        : outcome_(std::move(outcome))
        , state_(std::move(state))
        , label_(label)
    {
    }

    vector<Outcome>::const_iterator outcome_{};
    std::shared_ptr<FlatState> state_ = nullptr;
    size_t label_;
};

static_assert(std::forward_iterator<SuccessorIterator>);

class SuccessorRange : public std::ranges::view_interface<SuccessorRange> {
public:
    using iterator = SuccessorIterator;
    SuccessorRange() = default;

    [[nodiscard]]
    iterator begin() const
    {
        return begin_;
    }

    [[nodiscard]]
    iterator end() const
    {
        return end_;
    }

private:
    friend class OutcomeSuccessorGenerator;

    SuccessorRange(iterator begin, iterator end)
        : begin_(std::move(begin))
        , end_(std::move(end))
    {
    }

    iterator begin_{};
    iterator end_{};
};

static_assert(std::ranges::range<SuccessorRange>);

class OutcomeSuccessorGenerator {
public:
    explicit OutcomeSuccessorGenerator(const vector<Action>* actions)
        : actions_(actions)
    {
    }

    [[nodiscard]]
    SuccessorRange
    operator()(std::shared_ptr<FlatState> state, size_t action_index) const;

private:
    const vector<Action>* actions_;
};

class ApplicableActionsIterator {
public:
    using difference_type = int;
    using value_type = SuccessorRange;

    ApplicableActionsIterator() = default;

    [[nodiscard]]
    value_type operator*() const
    {
        return get_outcomes_(state_, aops_->at(pos_));
    }

    [[nodiscard]]
    const ApplicableActionsIterator* operator->() const
    {
        return this;
    }

    [[nodiscard]]
    SuccessorIterator begin() const
    {
        const auto& action = actions_->at(aops_->at(pos_));
        return SuccessorIterator(action.outcomes.begin(), state_, action.label);
    }

    [[nodiscard]]
    SuccessorIterator end() const
    {
        const auto& action = actions_->at(aops_->at(pos_));
        return SuccessorIterator(action.outcomes.end(), nullptr, action.label);
    }

    [[nodiscard]]
    size_t label() const
    {
        const auto& action = actions_->at(aops_->at(pos_));
        return action.label;
    }

    ApplicableActionsIterator& operator++()
    {
        ++pos_;
        return *this;
    }

    ApplicableActionsIterator operator++(int)
    {
        ApplicableActionsIterator temp(*this);
        ++pos_;
        return temp;
    }

    [[nodiscard]]
    bool operator==(const ApplicableActionsIterator& other) const
    {
        return pos_ == other.pos_;
    }

private:
    friend class ApplicableActionsRange;

    ApplicableActionsIterator(
        const vector<Action>* actions,
        std::shared_ptr<FlatState> state,
        std::shared_ptr<ApplicableActions> aops,
        size_t pos)
        : get_outcomes_(actions)
        , actions_(actions)
        , state_(std::move(state))
        , aops_(std::move(aops))
        , pos_(pos)
    {
    }

    OutcomeSuccessorGenerator get_outcomes_{nullptr};
    const vector<Action>* actions_;
    std::shared_ptr<FlatState> state_ = nullptr;
    std::shared_ptr<ApplicableActions> aops_ = nullptr;
    size_t pos_ = -1;
};

static_assert(std::forward_iterator<ApplicableActionsIterator>);

class ApplicableActionsRange
    : public std::ranges::view_interface<ApplicableActionsRange> {
public:
    using iterator = ApplicableActionsIterator;

    ApplicableActionsRange() = default;

    [[nodiscard]]
    iterator begin() const
    {
        return ApplicableActionsIterator(actions_, state_, aops_, 0u);
    }

    [[nodiscard]]
    iterator end() const
    {
        return ApplicableActionsIterator(
            actions_,
            state_,
            aops_,
            aops_ ? aops_->size() : 0ul);
    }

private:
    template <typename _ApplicableActionsGenerator>
    friend class SuccessorGenerator;

    ApplicableActionsRange(
        const vector<Action>* actions,
        std::shared_ptr<FlatState> state,
        std::shared_ptr<ApplicableActions> aops)
        : actions_(actions)
        , state_(std::move(state))
        , aops_(std::move(aops))
    {
    }

    const vector<Action>* actions_ = nullptr;
    std::shared_ptr<FlatState> state_ = nullptr;
    std::shared_ptr<ApplicableActions> aops_ = nullptr;
};

static_assert(std::ranges::range<ApplicableActionsRange>);

template <typename _ApplicableActionsGenerator>
class SuccessorGenerator {
public:
    explicit SuccessorGenerator(
        const vector<Action>* actions,
        _ApplicableActionsGenerator aops_gen)
        : aops_gen_(std::move(aops_gen))
        , actions_(actions)
    {
    }

    ApplicableActionsRange operator()(const FlatState& state)
    {
        return get_successors(std::make_shared<FlatState>(state));
    }

    ApplicableActionsRange operator()(const CompressedState& state)
    {
        return get_successors(std::make_shared<FlatState>(unpack(state)));
    }

protected:
    ApplicableActionsRange get_successors(std::shared_ptr<FlatState> state)
    {
        auto aops = std::make_shared<ApplicableActions>(aops_gen_(*state));
        return ApplicableActionsRange(
            actions_,
            std::move(state),
            std::move(aops));
    }

    _ApplicableActionsGenerator aops_gen_;
    const vector<Action>* actions_;
};

} // namespace police::successor_generator
