#pragma once

#include "police/storage/vector.hpp"

#include <functional>
#include <type_traits>

namespace police {

template <typename T = void>
struct hash : public std::hash<T> {};

namespace detail {
// https://www.boost.org/doc/libs/1_55_0/doc/html/hash/reference.html#boost.hash_combine
[[nodiscard]]
constexpr std::size_t hash_combine(std::size_t seed, std::size_t hash)
{
    return seed ^ (hash + 0x9e3779b9 + (seed << 6) + (seed >> 2));
}
} // namespace detail

template <typename... SizeT>
[[nodiscard]]
constexpr std::size_t
hash_combine(std::size_t seed, std::size_t hash, SizeT... ts)
{
    if constexpr (sizeof...(SizeT) == 0) {
        return detail::hash_combine(seed, hash);
    } else {
        return hash_combine(detail::hash_combine(seed, hash), ts...);
    }
}

template <typename Iterator, typename Sentinal>
[[nodiscard]]
constexpr std::size_t hash_sequence(Iterator first, Sentinal last)
{
    if (first == last) {
        return hash<std::size_t>()(0u);
    } else {
        using T = std::decay_t<decltype(*first)>;
        hash<T> hash_fn;
        std::size_t size = 1;
        std::size_t h = hash_fn(*first);
        for (++first; first != last; ++first, ++size) {
            h = hash_combine(h, hash_fn(*first));
        }
        return hash_combine(h, hash<std::size_t>()(size));
    }
}

template <typename T>
[[nodiscard]]
constexpr std::size_t hash_array(const T* base, std::size_t size)
{
    return hash_sequence(base, base + size);
}

template <>
struct hash<void> {
    template <typename T>
    [[nodiscard]]
    std::size_t operator()(T&& t) const
    {
        hash<std::decay_t<T>> h;
        return h(std::forward<T>(t));
    }
};

template <typename T1, typename T2>
struct hash<std::pair<T1, T2>> {
    [[nodiscard]]
    std::size_t operator()(const std::pair<T1, T2>& pr) const
    {
        return hash_combine(
            hash<std::remove_cv_t<T1>>()(pr.first),
            hash<std::remove_cv_t<T2>>()(pr.second));
    }
};

template <typename T>
struct hash<vector<T>> {
    std::size_t operator()(const vector<T>& vec) const
    {
        return hash_sequence(vec.begin(), vec.end());
    }
};

template <>
struct hash<vector<bool>> {
    [[nodiscard]]
    std::size_t operator()(const vector<bool>& vec) const
    {
        hash<std::size_t> hash_fn;
        std::size_t base_hash = hash_fn(vec.size());
        const std::size_t N = sizeof(std::size_t) * 8;
        for (std::size_t i = 0; i < vec.size();) {
            std::size_t v = 0;
            const std::size_t n = std::min(vec.size(), i + N);
            for (std::size_t b = 0; i < n; ++i, ++b) {
                v = v | (static_cast<std::size_t>(vec[i])) << b;
            }
            base_hash = hash_combine(base_hash, hash_fn(v));
        }
        return base_hash;
    }
};

template <typename... Args>
struct hash<std::tuple<Args...>> {
private:
    template <std::size_t Idx>
    void combine(std::size_t& h, const std::tuple<Args...>& args) const
    {
        if constexpr (Idx < sizeof...(Args)) {
            h = detail::hash_combine(
                h,
                hash<std::decay_t<decltype(std::get<Idx>(args))>>()(
                    std::get<Idx>(args)));
            combine<Idx + 1>(h, args);
        }
    }

public:
    std::size_t operator()(const std::tuple<Args...>& args) const
    {
        std::size_t h = hash<std::size_t>()(sizeof...(Args));
        combine<0>(h, args);
        return h;
    }
};

template <typename T>
[[nodiscard]]
std::size_t get_hash(T&& t)
{
    return hash<>()(std::forward<T>(t));
}

} // namespace police
