#include "police/utils/stopwatch.hpp"

#include <chrono>
#include <iostream>
#include <ratio>
#include <string_view>

namespace police {

Chronometer::Chronometer()
    : tstmp_(clock_type::now())
{
}

Chronometer::duration_type Chronometer::operator()() const
{
    return clock_type::now() - tstmp_;
}

StopWatch::StopWatch(bool start)
    : dur_(std::chrono::high_resolution_clock::duration::zero())
    , clock_()
    , running_(start)
{
}

Chronometer::duration_type StopWatch::get_duration() const
{
    if (running_) {
        return dur_ + clock_();
    } else {
        return dur_;
    }
}

void StopWatch::reset()
{
    dur_ = std::chrono::high_resolution_clock::duration::zero();
    clock_ = Chronometer();
    running_ = true;
}

void StopWatch::resume()
{
    if (!running_) {
        clock_ = Chronometer();
        running_ = true;
    }
}

void StopWatch::stop()
{
    if (running_) {
        running_ = false;
        dur_ += clock_();
    }
}

double StopWatch::get_milliseconds() const
{
    return std::chrono::duration_cast<
               std::chrono::duration<double, std::milli>>(get_duration())
        .count();
}

double StopWatch::get_seconds() const
{
    return std::chrono::duration_cast<
               std::chrono::duration<double, std::ratio<1>>>(get_duration())
        .count();
}

StopWatch::ScopedTimer StopWatch::scope()
{
    return ScopedTimer(this);
}

StopWatch::ScopedTimer::ScopedTimer(StopWatch* watch)
    : watch_(watch)
{
}

StopWatch::ScopedTimer::~ScopedTimer()
{
    watch_->dur_ += clock_();
}

StopWatch& StopWatch::operator+=(const StopWatch& other)
{
    dur_ += other.get_duration();
    return *this;
}

ScopedStopWatch::ScopedStopWatch(std::string_view name, bool start, Unit unit)
    : StopWatch(start)
    , name_(name)
    , unit_(unit)
{
}

void ScopedStopWatch::cancel()
{
    stop();
    destroyed_ = true;
}

void ScopedStopWatch::destroy()
{
    if (!destroyed_) {
        std::cout << name_ << ": ";
        switch (unit_) {
        case MILLISECONDS:
            std::cout << std::chrono::duration_cast<std::chrono::milliseconds>(
                get_duration());
            break;
        case SECONDS:
            std::cout << std::chrono::duration_cast<
                std::chrono::duration<double, std::ratio<1>>>(get_duration());
            break;
        default: std::cout << "unknown"; break;
        }
        std::cout << "\n";
        cancel();
    }
}

ScopedStopWatch::~ScopedStopWatch()
{
    destroy();
}

ScopedStopWatch& ScopedStopWatch::operator+=(const StopWatch& clock)
{
    StopWatch::operator+=(clock);
    return *this;
}

std::ostream& operator<<(std::ostream& out, const StopWatch& watch)
{
    return out << std::chrono::duration_cast<std::chrono::milliseconds>(
               watch.get_duration());
}

} // namespace police
