#pragma once

#include "police/nn_policy.hpp"
#include "police/execution_unit.hpp"

#include "police/smt_factory.hpp"
#include "police/variable_order_chooser.hpp"
#include "police/verifiers/ic3/sat_based/epic3_smt.hpp"
#include "police/verifiers/ic3/start_generator.hpp"

#include <memory>

namespace police {
struct Model;
struct VerificationProperty;
} // namespace police

namespace police::ic3 {

struct ReasonGeneralizerOption {
    enum Kind { GREEDY_MINIMIZE, UNSAT_CORE };
    virtual ~ReasonGeneralizerOption() = default;

    [[nodiscard]]
    virtual Kind kind() const = 0;
};

struct SatInterfaceOption {
    enum Kind { PER_LABEL, SINGLETON };
    virtual ~SatInterfaceOption() = default;
    [[nodiscard]]
    virtual Kind kind() const = 0;
};

struct GenMinimizeReason final : public ReasonGeneralizerOption {
    GenMinimizeReason(std::shared_ptr<VariableOrderChooser> order)
        : order(std::move(order))
    {
    }

    [[nodiscard]]
    Kind kind() const
    {
        return GREEDY_MINIMIZE;
    }

    std::shared_ptr<VariableOrderChooser> order;
};

struct GenUnsatCore final : public ReasonGeneralizerOption {
    GenUnsatCore() {}

    [[nodiscard]]
    Kind kind() const
    {
        return UNSAT_CORE;
    }
};

struct IC3Parameters {
    std::shared_ptr<ReasonGeneralizerOption> generalizer;
    std::shared_ptr<SatInterfaceOption> sat_interface;
    std::shared_ptr<SMTFactory> smt;
    bool obligation_rescheduling;
};

template <
    typename SuccessorGenerator,
    typename SatInterface,
    typename Generalizer>
class IC3Engine final : public ExecutionUnit {
public:
    using successor_generator_type = SuccessorGenerator;
    using start_generator_type = StartGenerator;
    using sat_interface_type = SatInterface;
    using reason_generalizer_type = Generalizer;
    using ic3_type = ExplicitStateIC3smt<
        successor_generator_type,
        sat_interface_type,
        reason_generalizer_type>;

    IC3Engine(
        std::shared_ptr<SMTFactory> smt_factory,
        const Model* model,
        const VerificationProperty* property,
        SuccessorGenerator successor_gen,
        ic3_type ic3);

    [[nodiscard]]
    std::string_view name() const override;

    void run() override;

    void report_statistics() override;

private:
    [[nodiscard]]
    bool some_start_is_goal() const;

    const Model* model_;
    const VerificationProperty* property_;
    SuccessorGenerator sgen_;
    std::shared_ptr<SMTFactory> smt_;
    ic3_type ic3_;
};

template <typename SuccessorGenerator>
std::shared_ptr<ExecutionUnit> create_ic3_engine(
    const IC3Parameters& params,
    const Model* model,
    const VerificationProperty* property,
    SuccessorGenerator&& successor_gen);

template <typename SuccessorGenerator>
std::shared_ptr<ExecutionUnit> create_ic3_engine(
    const IC3Parameters& params,
    const Model* model,
    const VerificationProperty* property,
    const NeuralNetworkPolicy& net,
    SuccessorGenerator&& successor_gen);

} // namespace police::ic3
