#pragma once

#include "police/base_types.hpp"
#include "police/storage/vector.hpp"
#include "police/verifiers/ic3/sat_based/sat_interface/nnlp.hpp"
#include <algorithm>

namespace police::ic3 {

template <typename BaseSat>
class MultiSatBase {
public:
    explicit MultiSatBase(vector<BaseSat> base)
        : base_(std::move(base))
    {
    }

    std::pair<bool, size_t> is_blocked(const Cube& cube, size_t frame)
    {
        size_t min_frame = -1;
        for (auto& sat : base_) {
            const auto b = sat.is_blocked(cube, frame);
            if (!b.first) {
                return {false, 0};
            }
            min_frame = std::min(min_frame, b.second);
        }
        return {true, min_frame};
    }

    void set_blocked(const Cube& cube, size_t frame)
    {
        for (auto& sat : base_) {
            sat.set_blocked(cube, frame);
        }
    }

    void add_frame()
    {
        for (auto& sat : base_) {
            sat.add_frame();
        }
    }

    void clear_frames()
    {
        for (auto& sat : base_) {
            sat.clear_frames();
        }
    }

    const vector<BaseSat>& base() const { return base_; }

protected:
    vector<BaseSat> base_;
};

template <typename BaseSat>
class MultiSat : public MultiSatBase<BaseSat> {};

template <>
class MultiSat<SatInterfaceNNLP> : public MultiSatBase<SatInterfaceNNLP> {
public:
    using MultiSatBase<SatInterfaceNNLP>::MultiSatBase;

    std::pair<bool, size_t> is_blocked(const Cube& cube, size_t frame);

    const Cube& get_unsat_core() const { return unsat_core_; }

    [[nodiscard]]
    bool supports_unsat_core() const
    {
        return std::all_of(base_.begin(), base_.end(), [](const auto& sat) {
            return sat.supports_unsat_core();
        });
    }

private:
    Cube unsat_core_;
};

} // namespace police::ic3
