#include "police/verifiers/ic3/syntactic/frame_refiner.hpp"

#include "police/action.hpp"
#include "police/linear_constraint.hpp"
#include "police/linear_expression.hpp"
#include "police/macros.hpp"
#include "police/storage/flat_state.hpp"
#include "police/storage/variable_space.hpp"
#include "police/utils/algorithms.hpp"
#include "police/verifiers/ic3/frame.hpp"
#include "police/verifiers/ic3/syntactic/abstraction.hpp"
#include "police/verifiers/ic3/syntactic/frames_storage.hpp"
#include "police/verifiers/ic3/syntactic/policy_reasoner.hpp"
#include "police/verifiers/ic3/syntactic/sufficient_condition.hpp"
#include "police/verifiers/ic3/syntactic/variable_classification.hpp"

#include <algorithm>
#include <cstdint>
#include <memory>
#include <utility>

#define VERBOSE_DEBUG_PRINTS 0

#if VERBOSE_DEBUG_PRINTS
#include "police/utils/io.hpp"

namespace police {
namespace {
struct print_suff_cond {
    const ic3::syntactic::SuffCondAlternatives* cond;
};

std::ostream& operator<<(std::ostream& out, const print_suff_cond& cond)
{
    out << "{";
    bool sep = false;
    for (const auto& x : *cond.cond) {
        out << (sep ? "; " : "") << print_sequence(x);
        sep = true;
    }
    out << "}";
    return out;
}

} // namespace
} // namespace police
#endif

namespace police::ic3::syntactic {
SyntacticFrameRefiner::Statistics::Statistics()
    : total_time(false)
    , policy_time(false)
    , update_time(false)
    , hitting_set_time(false)
{
}

namespace {
constexpr std::uint8_t INAPPLICABLE = 0;
constexpr std::uint8_t APPLICABLE = 1;
constexpr std::uint8_t RECHECK_GUARD = 2;
constexpr std::uint8_t RECHECK_POLICY = 4;
constexpr std::uint8_t POLICY_BLOCKED = 8;

#if VERBOSE_DEBUG_PRINTS
[[maybe_unused]]
const char* applicability_status_string(std::uint8_t status)
{
    switch (status) {
    case INAPPLICABLE: return "inapplicable";
    case APPLICABLE: return "applicable";
    case POLICY_BLOCKED: return "policy_blocked";
    }
    if (status & RECHECK_GUARD) return "recheck_guard";
    if (status & RECHECK_POLICY) return "recheck_policy";
    return "unknown";
}
#endif
} // namespace

SyntacticFrameRefiner::SyntacticFrameRefiner(
    FramesStorage* frames,
    std::shared_ptr<LinearCondition> goal,
    std::shared_ptr<LinearCondition> avoid,
    const Model* model,
    vector<size_t> ranked_vars,
    SyntacticAbstraction* abstraction,
    std::shared_ptr<PolicyReasoner> policy_reasoner,
    PolicyReasons* reasons)
    : ranked_vars_(std::move(ranked_vars))
    , frames_(frames)
    , abstraction_(abstraction)
    , model_(model)
    , goal_(goal)
    , avoid_(avoid)
    , policy_reasoner_(std::move(policy_reasoner))
    , reasons_(reasons)
    , var_class_(classify_variables(*model))
{
#ifndef POLICE_NO_STATISTICS
    stats_file_ = std::make_shared<std::ofstream>("synic3_refiner.stats");
#endif
    for (size_t var = 0; var < model->variables.size(); ++var) {
        const auto& type = model->variables[var].type;
        var_bounds_.set(
            var,
            Interval(type.get_lower_bound(), type.get_upper_bound()));
    }
}

namespace {
Cube propagate_tighten_bounds(
    const LinearConstraintConjunction& source,
    const Cube& var_bounds)
{
    Cube bounds(var_bounds);

    // propagate constraints until fixed point is reached
    bool changed = true;
    while (changed) {
        changed = false;
        for (const auto& constraint : source) {
            real_t lb = -constraint.rhs;
            real_t ub = -constraint.rhs;
            for (const auto& [var, coef] : constraint) {
                if (coef > 0.) {
                    lb += coef * static_cast<real_t>(bounds[var].second.lb);
                    ub += coef * static_cast<real_t>(bounds[var].second.ub);
                } else {
                    lb += coef * static_cast<real_t>(bounds[var].second.ub);
                    ub += coef * static_cast<real_t>(bounds[var].second.lb);
                }
            }
            for (const auto& [var, coef] : constraint) {
                Interval new_bound = bounds[var].second;
                real_t lb_prime = 0;
                real_t ub_prime = 0;
                if (coef > 0.) {
                    lb_prime = (ub - coef * static_cast<real_t>(
                                                bounds[var].second.ub)) /
                               -coef;
                    ub_prime = (lb - coef * static_cast<real_t>(
                                                bounds[var].second.lb)) /
                               -coef;
                } else {
                    lb_prime = (lb - coef * static_cast<real_t>(
                                                bounds[var].second.ub)) /
                               -coef;
                    ub_prime = (ub - coef * static_cast<real_t>(
                                                bounds[var].second.lb)) /
                               -coef;
                }
                switch (constraint.type) {
                case LinearConstraint::EQUAL:
                    new_bound.tighten(Value(lb_prime), Value(ub_prime));
                    break;
                case LinearConstraint::LESS_EQUAL:
                    new_bound.tighten(new_bound.lb, Value(ub_prime));
                    break;
                case LinearConstraint::GREATER_EQUAL:
                    new_bound.tighten(Value(lb_prime), new_bound.ub);
                    break;
                }
                if (new_bound != bounds[var].second) {
                    bounds[var].second = new_bound;
                    changed = true;
                }
            }
        }
    }

    return bounds;
}

void remove_implied_conditions(
    SufficientCondition& dest,
    const flat_state& state,
    const Cube& bounds)
{
    size_t i = 0;
    for (size_t j = 0; j < dest.size(); ++j) {
        bool implied = false;
        size_t var = dest[j].variable_id;
        const Interval& interval = bounds[var].second;
        switch (dest[j].type) {
        case VariableCondition::EQUALITY:
            implied = interval.lb == interval.ub && interval.ub == state[var];
            break;
        case VariableCondition::UPPER_BOUND:
            implied = interval.ub <= state[var];
            break;
        case VariableCondition::LOWER_BOUND:
            implied = interval.lb >= state[var];
            break;
        }
        if (!implied) {
            dest[i] = dest[j];
            ++i;
        }
    }
    dest.erase(dest.begin() + i, dest.end());
}

void remove_implied_conditions(
    SuffCondAlternatives& dest,
    const flat_state& state,
    const LinearConstraintConjunction& source,
    const Cube& var_bounds)
{
    // std::cout << "REMOVING REDUNDANT INFORMATION" << "\n";
    // std::cout << source << std::endl;
    const Cube bounds = propagate_tighten_bounds(source, var_bounds);
    for (auto& x : dest) {
        // std::cout << "in:  " << print_sequence(x) << std::endl;
        remove_implied_conditions(x, state, bounds);
        // std::cout << "out: " << print_sequence(x) << std::endl;
    }
}

void make_variable_conditions_unique(
    size_t num_vars,
    SufficientCondition& result)
{
    vector<int> k(num_vars, -1);
    for (const auto& cond : result) {
        if (k[cond.variable_id]) {
            k[cond.variable_id] = cond.type;
        } else if (
            cond.type == VariableCondition::EQUALITY ||
            k[cond.variable_id] != cond.type) {
            k[cond.variable_id] = VariableCondition::EQUALITY;
        }
    }
    SufficientCondition ordered;
    for (size_t var = 0; var < k.size(); ++var) {
        if (k[var] >= 0) {
            ordered.emplace_back(var, VariableCondition::Type(k[var]));
        }
    }
    result.swap(ordered);
    // std::cout << "in  " << print_sequence(ordered) << std::endl;
    // std::cout << "out " << print_sequence(result) << std::endl;
}

[[maybe_unused]]
vector<size_t> get_index_map_collapse(
    const vector<size_t>& var_order,
    vector<SuffCondAlternatives>& options)
{
    vector<size_t> map(var_order.size() * 3, 0);
    // phase 1: collect all conditions
    for (const auto& option : options) {
        for (const auto& cand : option) {
            for (const auto& crit : cand) {
                map[crit.index()] = 1;
            }
        }
    }
    // phase 2: assign ids
    vector<VariableCondition> strongest;
    size_t id = 0;
    for (const size_t var : var_order) {
        const size_t index = 3 * var + 2;
        if (map[index] || (map[index - 1] && map[index - 2])) {
            strongest.push_back(
                VariableCondition(var, VariableCondition::EQUALITY));
            map[index] = id;
            map[index - 1] = id;
            map[index - 2] = id;
            ++id;
        } else {
            if (map[index - 1]) {
                strongest.push_back(
                    VariableCondition(var, VariableCondition::UPPER_BOUND));
                map[index - 1] = id;
                ++id;
            }
            if (map[index - 2]) {
                strongest.push_back(
                    VariableCondition(var, VariableCondition::LOWER_BOUND));
                map[index - 2] = id;
                ++id;
            }
        }
    }
    // phase 3: for each var, only keep strongest condition
    vector<bool> duplicate(id, false);
    for (auto& option : options) {
        for (auto& cand : option) {
            std::fill(duplicate.begin(), duplicate.end(), false);
            size_t j = 0;
            for (size_t i = 0; i < cand.size(); ++i) {
                const auto idx = map[cand[i].index()];
                if (!duplicate[idx]) {
                    duplicate[idx] = true;
                    cand[j] = strongest[idx];
                    ++j;
                }
            }
            cand.erase(cand.begin() + j, cand.end());
        }
    }
    return map;
}

[[maybe_unused]]
vector<size_t> get_index_map_separate(
    const vector<size_t>& var_order,
    vector<SuffCondAlternatives>& options)
{
    vector<size_t> map(var_order.size() * 3, -1);
    // phase 1: collect all conditions, replace EQUALITY by UPPER and LOWER
    // bounds
    for (auto& option : options) {
        for (auto& cand : option) {
            for (int i = cand.size() - 1; i >= 0; --i) {
                VariableCondition& cond = cand[i];
                if (cond.type == VariableCondition::EQUALITY) {
                    cond.type = VariableCondition::UPPER_BOUND;
                    map[cond.index()] = 1;
                    cand.push_back(VariableCondition(
                        cond.variable_id,
                        VariableCondition::LOWER_BOUND));
                    map[cand.back().index()] = 1;
                } else {
                    map[cond.index()] = 1;
                }
            }
        }
    }
    // phase 2: assign ids
    size_t id = 0;
    for (const size_t var : var_order) {
        const size_t index = 3 * var;
        POLICE_ASSERT(map[index + 2] == (size_t)-1);
        if (map[index] == 1) {
            map[index] = id;
            ++id;
        }
        if (map[index + 1] == 1) {
            map[index + 1] = id;
            ++id;
        }
    }
    return map;
}

SufficientCondition get_hitting_set(
    const vector<size_t>& var_order,
    vector<SuffCondAlternatives>& options)
{
#if VERBOSE_DEBUG_PRINTS
    // std::cout << "finding hitting set:" << std::endl;
    // for (const auto& alt : options) {
    //     std::cout << print_suff_cond(&alt) << std::endl;
    // }
#endif
    auto ids = get_index_map_collapse(var_order, options);
    auto result =
        greedy_hitting_set(options, [&ids](const VariableCondition& cond) {
            POLICE_ASSERT(cond.index() < ids.size());
            return ids[cond.index()];
        });
    // make_variable_conditions_unique(var_order.size(), result);
    return result;
}

} // namespace

size_t
SyntacticFrameRefiner::add_reason(const flat_state& state, size_t target_frame)
{
#if VERBOSE_DEBUG_PRINTS
    std::cout << "frame refinement for state " << print_sequence(state)
              << " at frame " << target_frame << std::endl;
#endif

    auto sw = stats_.total_time.scope();

    vector<SuffCondAlternatives> options;
    vector<std::uint8_t> applicable(model_->actions.size(), INAPPLICABLE);

    // make sure that avoid condition is in contradiction
    options.push_back(get_suff_cond_violated(state, *avoid_));
    POLICE_ASSERT(!options.back().empty());

    // check if goal is satisfied
    {
        auto set = get_suff_cond_satisfied(state, *goal_);
        if (!set.empty()) {
            // goal state

            ++stats_.no_policy_reason;
            ++stats_.goal_reason;

            drop_trivial_conditions(set, state);
            POLICE_ASSERT(!set.empty());
            options.push_back(std::move(set));

#if VERBOSE_DEBUG_PRINTS
            std::cout << "  => OPTIONS = " << print_suff_cond(&options.back())
                      << std::endl;
#endif

            // choose a subset of variable that hits all options
            const auto vset = stats_.hitting_set_time.function_call(
                [&](auto& options) {
                    return get_hitting_set(ranked_vars_, options);
                },
                options);

#ifndef POLICE_NO_STATISTICS
            (*stats_file_) << "{" << "\"frame\": " << target_frame << ", "
                           << "\"cutoff\": true, "
                           << "\"policy\":" << 0 << ", "
                           << "\"size\":" << vset.size()
                           << ", \"vars\": " << print_sequence(vset) << "}\n";
#endif

            return update_frames(
                applicable,
                project_state(state, vset),
                CubeDatabase::INF_FRAME);
        }
    }

    options.reserve(options.size() + model_->actions.size());

    // prepare policy reasoner
    if (policy_reasoner_) {
        policy_reasoner_->prepare(state);
    }

    // handle each action
    size_t i = 0;
    auto it = model_->actions.begin();
    for (; i < model_->actions.size() && it->label == SILENT_ACTION;
         ++i, ++it) {
        applicable[i] = handle_silent_action(options, state, *it, target_frame);
#if VERBOSE_DEBUG_PRINTS
        std::cout << "  => OPTIONS = " << print_suff_cond(&options.back())
                  << std::endl;
#endif
    }
    assert(
        std::is_sorted(
            it,
            model_->actions.end(),
            [](const auto& a, const auto& b) { return a.label < b.label; }));
    bool policy_reason = false;
    unsigned long long policy_calls = 0;
    for (size_t label = 0; label < model_->labels.size(); ++label) {
        POLICE_ASSERT(
            i < model_->actions.size() && model_->actions[i].label == label);
        const bool x = handle_labeled_actions(
            applicable,
            options,
            state,
            label,
            i,
            target_frame);
        policy_reason |= x > 0;
        policy_calls += x;
    }

    stats_.policy_reason += policy_reason;
    stats_.no_policy_reason += !policy_reason;
    stats_.policy_calls_max = std::max(policy_calls, stats_.policy_calls_max);

    // choose a subset of variable that hits all options
#if VERBOSE_DEBUG_PRINTS
    std::cout << "- finding hitting set\n";
    for (const auto& x : options) {
        std::cout << "  " << print_suff_cond(&x) << "\n";
    }
    std::cout << "===> " << std::flush;
#endif
    const auto vset = stats_.hitting_set_time.function_call(
        [&](auto& options) { return get_hitting_set(ranked_vars_, options); },
        options);
#if VERBOSE_DEBUG_PRINTS
    std::cout << print_sequence(vset) << std::endl;
#endif

#ifndef POLICE_NO_STATISTICS
    (*stats_file_) << "{" << "\"frame\": " << target_frame << ", "
                   << "\"cutoff\": false, "
                   << "\"policy\":" << policy_calls << ", "
                   << "\"size\":" << vset.size()
                   << ", \"vars\": " << print_sequence(vset) << "}\n";
#endif

    // choose a subset of variables covering all options and add the
    // corresponding cube to the frame
    return update_frames(applicable, project_state(state, vset), target_frame);
}

std::uint8_t SyntacticFrameRefiner::handle_silent_action(
    vector<SuffCondAlternatives>& reasons,
    const flat_state& state,
    const Action& action,
    size_t frame_idx) const
{
    // first check guard
    vector<std::pair<size_t, Interval>> relaxed_data;
    relaxed_data.reserve(var_class_.size());
    for (size_t var = 0; var < var_class_.size(); ++var) {
        relaxed_data.emplace_back(var, Interval(state[var]));
    }
    Cube relaxed_state(std::move(relaxed_data));
    SuffCondAlternatives set =
        get_suff_cond_violated(state, action.guard, relaxed_state);

    std::uint8_t check_guard = 0;

    if (!set.empty()) {
        drop_trivial_conditions(set, state);
        POLICE_ASSERT(!set.empty());
        // add precondition back into relaxed state
        apply_unit_constraints(relaxed_state, action.guard);
        // (don't do constraint propagation, which is currently not supported by
        // the regression computation)
        check_guard = RECHECK_GUARD;
    }

    const size_t n = reasons.size();
    // if guard is satisfied, generate variable set candidates for each outcome,
    // guaranteeing that the transition results in a cube of the desired frame
    for (const auto& outcome : action.outcomes) {
        if (!handle_outcome(
                reasons,
                state,
                relaxed_state,
                action,
                outcome.assignments,
                frame_idx)) {
            POLICE_ASSERT(!set.empty());
            reasons.resize(n);
            reasons.push_back(std::move(set));
            return INAPPLICABLE;
        } else {
            auto& r = reasons.back();
            r.insert(r.end(), set.begin(), set.end());
        }
    }

    return APPLICABLE | check_guard;
}

bool SyntacticFrameRefiner::handle_labeled_actions(
    vector<std::uint8_t>& applicability,
    vector<SuffCondAlternatives>& reasons,
    const flat_state& state,
    size_t label,
    size_t& action_idx,
    size_t frame_idx) const
{
    const size_t last_action_idx = action_idx;
    const size_t options_start = reasons.size();
    auto it = model_->actions.begin() + action_idx;
    for (; action_idx < model_->actions.size() && it->label == label;
         ++action_idx, ++it) {
        const size_t old_options = reasons.size();
        const std::uint8_t status =
            handle_labeled_action(reasons, state, *it, frame_idx);
        if (status == POLICY_BLOCKED) {
            reasons.erase(
                reasons.begin() + options_start,
                reasons.begin() + old_options);
            POLICE_ASSERT(reasons.size() == options_start + 1u);
#if VERBOSE_DEBUG_PRINTS
            std::cout << " - policy doesn't select " << it->label << " if "
                      << print_suff_cond(&reasons.back()) << std::endl;
#endif
            for (size_t i = last_action_idx; i < action_idx; ++i) {
                applicability[i] = INAPPLICABLE;
            }
            for (; action_idx < model_->actions.size() && it->label == label;
                 ++action_idx, ++it) {
                applicability[action_idx] = INAPPLICABLE;
            }
            SuffCondAlternatives policy_reasons(std::move(reasons.back()));
            reasons.pop_back();
            for (size_t i = last_action_idx; i < action_idx; ++i) {
                reasons.push_back(policy_reasons);
                remove_implied_conditions(
                    reasons.back(),
                    state,
                    model_->actions[i].guard,
                    var_bounds_);
#if VERBOSE_DEBUG_PRINTS
                std::cout << "  => OPTIONS = "
                          << print_suff_cond(&reasons.back()) << " (action "
                          << i << ")" << std::endl;
#endif
            }
            return true;
        } else {
            applicability[action_idx] = status;
#if VERBOSE_DEBUG_PRINTS
            std::cout << "  => OPTIONS = " << print_suff_cond(&reasons.back())
                      << std::endl;
#endif
        }
    }
    return false;
}

std::uint8_t SyntacticFrameRefiner::handle_labeled_action(
    vector<SuffCondAlternatives>& reasons,
    const flat_state& state,
    const Action& action,
    size_t frame_idx) const
{
    // similar to silent actions except that we here also need to take care of
    // the policy selections
    // start with guard
    vector<std::pair<size_t, Interval>> relaxed_data;
    relaxed_data.reserve(var_class_.size());
    for (size_t var = 0; var < var_class_.size(); ++var) {
        relaxed_data.emplace_back(var, Interval(state[var]));
    }
    Cube relaxed_state(std::move(relaxed_data));
    SuffCondAlternatives set =
        get_suff_cond_violated(state, action.guard, relaxed_state);

    std::uint8_t status_flags = 0;

    assert(set.empty() == action.guard.evaluate([&state](size_t idx) {
        return static_cast<real_t>(state[idx]);
    }));

    if (!set.empty()) {
        drop_trivial_conditions(set, state);
        POLICE_ASSERT(!set.empty());
        // add precondition back into relaxed state
        apply_unit_constraints(relaxed_state, action.guard);
        // (don't do constraint propagation, which is currently not supported by
        // the regression computation)
        status_flags = RECHECK_GUARD;
    }

#if VERBOSE_DEBUG_PRINTS
    std::cout << "- handling action at frame " << frame_idx
              << ": id=" << (&action - &model_->actions.front())
              << " label_id=" << action.label << " ("
              << model_->labels[action.label] << ")"
              << " applicable=" << set.empty();
#endif

    if (reasons_) {
        const vector<Cube> blockers =
            reasons_->get_reasons(state, action.label);
#if VERBOSE_DEBUG_PRINTS
        std::cout << " blocked=" << (!blockers.empty());
#endif
        if (!blockers.empty()) {
            for (const auto& cube : blockers) {
                SufficientCondition cond;
                for (const auto& [var, bounds] : cube) {
                    bool lb = bounds.lb > var_bounds_[var].second.lb;
                    bool ub = bounds.ub < var_bounds_[var].second.ub;
                    if (lb || ub) {
                        cond.push_back(VariableCondition::make(var, lb, ub));
                    }
                }
                set.push_back(std::move(cond));
            }
            drop_trivial_conditions(set, state);
            POLICE_ASSERT(!set.empty());
            remove_implied_conditions(set, state, action.guard, var_bounds_);
            status_flags = status_flags | RECHECK_POLICY;
        }
    }

#if VERBOSE_DEBUG_PRINTS
    std::cout << "\n  -> relaxed_state=" << relaxed_state << std::endl;
#endif

    // check if all outcomes are inserted (if so, policy choice doesn't matter)
    const size_t old_reasons = reasons.size();
    bool outcomes_handled = true;
    for (const auto& outcome : action.outcomes) {
        if (!handle_outcome(
                reasons,
                state,
                relaxed_state,
                action,
                outcome.assignments,
                frame_idx)) {
            outcomes_handled = false;
            // revert options and get a reason of why the policy doesn't select
            // the considered action
            reasons.erase(reasons.begin() + old_reasons, reasons.end());
#if VERBOSE_DEBUG_PRINTS
            std::cout << "  |-> outcome's successor is in frame "
                      << (frame_idx - 1)
                      << " => action must be blocked by policy" << std::endl;
#endif
            break;
        } else {
            auto& r = reasons.back();
            r.insert(r.end(), set.begin(), set.end());
        }
    }

    if (!outcomes_handled) {
        if (set.empty()) {
#if VERBOSE_DEBUG_PRINTS
            std::cout << "  |- querying for policy reason" << std::endl;
#endif
            ++stats_.policy_calls_total;
            POLICE_ASSERT(policy_reasoner_ != nullptr);
            auto sw = stats_.policy_time.scope();
            auto r =
                policy_reasoner_->get_reason(state, action.guard, action.label);
            drop_trivial_conditions(r.back(), state);
            if (reasons_) {
                reasons_->block(project_state(state, r[0]), action.label);
            }
            POLICE_ASSERT(!r.empty());
#if VERBOSE_DEBUG_PRINTS
            std::cout << "  |-> policy reasoner returned: "
                      << print_sequence(r[0]) << std::endl;
#endif
            drop_trivial_conditions(r, state);
            r.insert(r.end(), set.begin(), set.end());
            reasons.push_back(std::move(r));
            return POLICY_BLOCKED | status_flags;
        } else {
            reasons.push_back(std::move(set));
            return (status_flags & RECHECK_GUARD) ? INAPPLICABLE
                                                  : POLICY_BLOCKED;
        }
    }

    return APPLICABLE | status_flags;
}

bool SyntacticFrameRefiner::handle_outcome(
    vector<SuffCondAlternatives>& reasons,
    const flat_state& state,
    const Cube& relaxed_state,
    const Action& action,
    const vector<Assignment>& outcome,
    size_t frame_idx) const
{
    POLICE_ASSERT(frame_idx >= 1u);
    POLICE_ASSERT(relaxed_state.size() == var_bounds_.size());

    const auto succ = get_post_condition(relaxed_state, outcome);

#if VERBOSE_DEBUG_PRINTS
    std::cout << "  |- outcome -> succ=" << succ << std::endl;
#endif

    SuffCondAlternatives rs;

    // special handling of last frame -> make sure successor violates the avoid
    // condition
    if (frame_idx == 1u) {
        vector<Cube> options = get_suff_cond_violated(succ, *avoid_);
        if (options.empty()) {
#if VERBOSE_DEBUG_PRINTS
            std::cout << "   |- successor doesn't satisfy avoid condition"
                      << std::endl;
#endif
            // successor satisfies avoid
            return false;
        }
#if VERBOSE_DEBUG_PRINTS
        std::cout << "   |- successor satisfies avoid condition:" << std::endl;
#endif
        for (auto& cube : options) {
#if VERBOSE_DEBUG_PRINTS
            std::cout << "    |- candidate cube: " << cube << "\n";
#endif
            drop_trivial_bounds(cube);
            POLICE_ASSERT(!cube.empty());
            rs.push_back(regress_cube(relaxed_state, cube, action, outcome));
        }
    } else {
        // get all satisfied cubes in frames >= frame_idx - 1
        const auto cubes = get_satisfied_cubes(succ, frame_idx - 1);
        if (cubes.empty()) {
#if VERBOSE_DEBUG_PRINTS
            std::cout << "   |- successor in frame " << (frame_idx - 1)
                      << std::endl;
#endif
            // transition is not inserted by current frames
            return false;
        }
#if VERBOSE_DEBUG_PRINTS
        std::cout << "   |- successor blocked from frame " << (frame_idx - 1)
                  << " (" << cubes.size() << " blocking cubes)"
                  << "\n";
#endif
        // for each cube, compute variables which retaining guarantees to
        // transition into that cube
        rs.reserve(cubes.size());
        for (const auto& cube_id : cubes) {
#if VERBOSE_DEBUG_PRINTS
            std::cout << "    |- candidate cube: " << cube_id << std::flush;
#endif
            const auto& cube = frames_->at(cube_id);
#if VERBOSE_DEBUG_PRINTS
            std::cout << " (" << cube << ")" << "\n";
#endif
            rs.push_back(regress_cube(relaxed_state, cube, action, outcome));
#if VERBOSE_DEBUG_PRINTS
            std::cout << "      |- regression: " << print_sequence(rs.back())
                      << std::endl;
#endif
        }
    }

    drop_trivial_conditions(rs, state);

    auto drop_duplicates = [](SuffCondAlternatives& conds) {
#if 0
        std::cout << "drop_duplicates conds.size() = " << conds.size() << std::endl;
        // remove duplicates
        std::sort(conds.begin(), conds.end(), [&](const auto& x, const auto& y) {
            std::cout << " compare " << (&x - &conds[0]) << " with " << (&y - &conds[0]) << std::endl;
            if (x.size() > y.size()) {
                return false;
            }
            if (x.size() < y.size()) {
                return true;
            }
            for (auto i = 0u; i < x.size(); ++i) {
                if (x[i] < y[i]) {
                    return true;
                } else if (y[i] < x[i]) {
                    return false;
                }
            }
            return false;
        });
        conds.erase(std::unique(conds.begin(), conds.end()), conds.end());
#else
        unordered_set<SufficientCondition> set;
        size_t n = 0;
        for (size_t i = 0; i < conds.size(); ++i) {
            if (set.insert(conds[i]).second) {
                if (n != i) {
                    conds[n] = std::move(conds[i]);
                }
                ++n;
            }
        }
        conds.erase(conds.begin() + n, conds.end());
#endif
    };

    drop_duplicates(rs);

    // store options
    rs.shrink_to_fit();
    reasons.push_back(std::move(rs));

    return true;
}

SufficientCondition SyntacticFrameRefiner::regress_cube(
    const Cube&,
    const Cube& cube,
    const Action& action,
    const vector<Assignment>& assignments) const
{
    assert(
        std::is_sorted(
            assignments.begin(),
            assignments.end(),
            [](const Assignment& o, const Assignment& oo) {
                return o.var_id < oo.var_id;
            }));

    vector<int> condition(var_bounds_.size(), -1);
    auto need_lb = [&](size_t var) -> size_t {
        return (condition[var] >= 0) && (condition[var] != 1);
    };
    auto need_ub = [&](size_t var) -> size_t { return (condition[var] >= 1); };
    auto mark_as_needed = [&](size_t var, size_t lb, size_t ub) {
        condition[var] =
            VariableCondition::get_type(lb | need_lb(var), ub | need_ub(var));
    };

    vector<bool> in_pre(var_bounds_.size(), false);
    for (const auto& cond : action.guard) {
        if (cond.size() == 1 && cond.type == LinearConstraint::EQUAL) {
            POLICE_ASSERT(cond.begin()->first < in_pre.size());
            in_pre[cond.begin()->first] = true;
        }
    }

    // #if VERBOSE_DEBUG_PRINTS
    //     std::cout << "    regress : " << std::endl;
    // #endif

    // collect all variables appearing in assignments x := phi where
    // x >= lb or x <= ub is in cube; in the former case, mark the HIGH half
    // of the variables in phi, in the latter case the LOW half
    size_t i = 0;
    for (auto it = cube.begin(); it != cube.end(); ++it) {
        // which bounds are needed?
        const bool need_lower_bound =
            it->second.lb > var_bounds_[it->first].second.lb;
        const bool need_upper_bound =
            it->second.ub < var_bounds_[it->first].second.ub;

        POLICE_ASSERT(need_lower_bound || need_upper_bound);

        // #if VERBOSE_DEBUG_PRINTS
        //         std::cout << "        want var#" << it->first << " in " <<
        //         (it->second)
        //                   << " => lb=" << need_lower_bound << " ub=" <<
        //                   need_upper_bound
        //                   << std::endl;
        // #endif

        // move to next assignment
        for (; i < assignments.size() && assignments[i].var_id < it->first;
             ++i) {
        }

        // #if VERBOSE_DEBUG_PRINTS
        //         std::cout << "          -> assignment: ";
        //         if (i < assignments.size() && assignments[i].var_id ==
        //         it->first) {
        //             std::cout << assignments[i].value << std::endl;
        //         } else {
        //             std::cout << "PREVAIL (in_pre: " << in_pre[it->first] <<
        //             ")"
        //                       << std::endl;
        //         }
        // #endif

        // check if next assignment corresponds to currently consider var
        if (i < assignments.size() && assignments[i].var_id == it->first) {
            // var assignment
            for (const auto& [var, coef] : assignments[i].value) {
                if (in_pre[var]) {
                    continue;
                }
                mark_as_needed(
                    var,
                    ((coef > 0. && need_lower_bound) ||
                     (coef < 0. && need_upper_bound)),
                    ((coef < 0. && need_lower_bound) ||
                     (coef > 0. && need_upper_bound)));
            }
        } else {
            // var prevails its value
            if (in_pre[it->first]) {
                continue;
            }
            mark_as_needed(it->first, need_lower_bound, need_upper_bound);
        }
    }

    vector<VariableCondition> result;
    for (size_t var = 0; var < condition.size(); ++var) {
        if (condition[var] >= 0) {
            result.emplace_back(var, VariableCondition::Type(condition[var]));
        }
    }

    // #if VERBOSE_DEBUG_PRINTS
    //     std::cout << "    result: " << print_sequence(result) << std::endl;
    // #endif

    return result;
}

void SyntacticFrameRefiner::apply_unit_constraints(
    Cube& cube,
    const LinearConstraintConjunction& guard) const
{
    // enrich cube with assignments implied by the action's guard
    for (const auto& cond : guard) {
        if (cond.size() == 1) {
            POLICE_ASSERT(cond.begin()->second == 1.);
            const size_t var = cond.begin()->first;
            Interval cur =
                cube.has(var) ? cube.get(var) : var_bounds_[var].second;
            const auto type =
                model_->variables.get_type(cond.begin()->first).value_type();
            if (cond.type == LinearConstraint::EQUAL) {
                cur = Value(cond.rhs, type);
                cube.set(var, cur);
            } else if (cond.type == LinearConstraint::LESS_EQUAL) {
                if (cond.rhs < cur.ub) {
                    cur.ub = Value(cond.rhs, type);
                    cube.set(var, cur);
                }
            } else {
                if (cond.rhs > cur.lb) {
                    cur.lb = Value(cond.rhs, type);
                    cube.set(var, cur);
                }
            }
        }
    }
}

void SyntacticFrameRefiner::propagate_constraints(
    Cube& cube,
    const LinearConstraintConjunction& guard) const
{
    Cube bounds(var_bounds_);
    for (const auto& [var, bnds] : cube) {
        bounds.shrink(var, bnds);
    }
    propagate_tighten_bounds(guard, bounds);
    for (auto& [var, bnds] : cube) {
        const auto& new_bnds = bounds[var].second;
        bnds.tighten(new_bnds.lb, new_bnds.ub);
    }
}

Cube SyntacticFrameRefiner::get_post_condition(
    const Cube& cube,
    const vector<Assignment>& outcomes) const
{
    POLICE_ASSERT(cube.size() == var_bounds_.size());

    // compute post condition
    Cube post(cube);
    for (const auto& ass : outcomes) {
        auto bnds = bound_combination(ass.value, cube);
        Interval& cur = post[ass.var_id].second;
        auto type = model_->variables[ass.var_id].type.value_type();
        cur = var_bounds_[ass.var_id].second;
        cur.tighten(
            Value(bnds.lb + ass.value.bias, type),
            Value(bnds.ub + ass.value.bias, type));
    }

    // remove vars with full domain
    for (auto it = post.begin(); it != post.end();) {
        if (it->second == var_bounds_[it->first].second) {
            it = post.erase(it);
        } else {
            ++it;
        }
    }

    return post;
}

size_t SyntacticFrameRefiner::update_frames(
    const vector<std::uint8_t>& applicable,
    const Cube& reason,
    [[maybe_unused]] size_t frame)
{
    auto sw = stats_.update_time.scope();

    // insert reason
    const auto [cube_id, new_cube] = register_reason(reason);

    POLICE_DEBUG_MSG(
        "reason: "
        << reason.dump(model_->variables) << " ratio="
        << ((static_cast<double>(reason.size()) / var_bounds_.size()))
        << " cube_id=" << cube_id << " is_new=" << new_cube << " cur_h="
        << (new_cube ? 0u : abstraction_->get_h_value(cube_id)) << "\n")

#ifndef NDEBUG
    for (auto [var, vals] : reason) {
        assert(
            vals.lb > var_bounds_[var].second.lb ||
            vals.ub < var_bounds_[var].second.ub);
    }
#endif

    [[maybe_unused]]
    size_t inserted_frame = 0;
    // if it is a new cube, update abstraction
    if (new_cube) {
        // update abstraction and get actual insertion point
        inserted_frame = update_abstraction(applicable, reason, cube_id);
    }
    // otherwise, compute new h value
    else {
        POLICE_ASSERT(frame > abstraction_->get_h_value(cube_id));
        // prune transitions for which we know are inapplicable (primarily
        // pertains to presence of policy; during previous generation of this
        // cube, might have treated some actions as being applicable as we did
        // not have to prove that that action is never selected by the policy)
        // with current implementation (always checking actual action
        // applicability before anything else), should happen only due to
        // policy not selecting an action
        auto& arcs = abstraction_->get_node_arcs(cube_id);
        size_t i = 0;
        for (size_t j = 0; j < arcs.size(); ++j) {
            const auto& arc = abstraction_->get_arc(arcs[j]);
            const auto idx = arc.action_idx;
#if VERBOSE_DEBUG_PRINTS
            std::cout << "arc#" << j << ": " << cube_id << " ->";
            for (auto x : arc.successors) {
                std::cout << " " << x << " (" << abstraction_->get_h_value(x)
                          << ")";
            }
            std::cout << " LABEL=" << arc.label << " INDEX=" << idx
                      << " APPLICABLE="
                      << applicability_status_string(applicable[idx]) << "("
                      << static_cast<int>(applicable[idx]) << ")" << std::endl;
#endif
            bool is_appl = static_cast<bool>(applicable[idx] & APPLICABLE);
            if (applicable[idx] & RECHECK_POLICY) {
                const auto& action = model_->actions[idx];
                POLICE_ASSERT(!is_conjunction_violated(action.guard, reason));
                POLICE_ASSERT(reasons_ != nullptr);
                auto context_cube = prepare_progression(reason, action.guard);
                is_appl = !reasons_->is_blocked(context_cube, action.label);
            }
            if (is_appl) {
                if (i != j) {
                    arcs[i] = std::move(arcs[j]);
                }
                ++i;
            } else {
                abstraction_->mark_arc_pruned(arc.arc_id);
            }
        }
        arcs.erase(arcs.begin() + i, arcs.end());
        inserted_frame = abstraction_->recompute_h_value(cube_id);
    }
    POLICE_ASSERT(inserted_frame >= frame);

    POLICE_DEBUG_MSG(
        "inserted cube " << cube_id << " at frame " << inserted_frame << "\n")

#ifndef NDEBUG
    for (size_t i = 0; i < frames_->num_cubes(); ++i) {
        if (i == cube_id) {
            continue;
        }
        const auto& i_arcs = abstraction_->get_in_arcs(i);
        const auto& j_arcs = abstraction_->get_in_arcs(cube_id);
        if (frames_->at(cube_id) <= frames_->at(i)) {
            for (const auto& arc_id : j_arcs) {
                assert(std::count(j_arcs.begin(), j_arcs.end(), arc_id));
            }
        } else if (frames_->at(i) <= frames_->at(cube_id)) {
            for (const auto& arc_id : i_arcs) {
                assert(std::count(i_arcs.begin(), i_arcs.end(), arc_id));
            }
        }
    }
#endif

    return cube_id;
}

size_t SyntacticFrameRefiner::insert_transition(
    size_t cube_id,
    const Cube& cube,
    const Cube& context_cube,
    size_t action_idx,
    const Action& action,
    const vector<Assignment>& outcomes)
{
    auto post = get_post_condition(context_cube, outcomes);

    // discard self-loop transitions
    if (post <= cube) {
        return CubeDatabase::INF_FRAME;
    }

    auto successors = get_implied_cubes(post);

    size_t frame = 0;
    for (const auto& succ : successors) {
        frame = std::max(frame, abstraction_->get_h_value(succ));
    }

#if VERBOSE_DEBUG_PRINTS
    std::cout << "arc " << cube_id << " -" << action.label << "-> "
              << print_sequence(successors) << " => " << frame
              << " idx=" << action_idx << " (" << post << ")" << std::endl;
#endif

    abstraction_->add_arc(
        cube_id,
        HyperArc(std::move(successors), action.label, action_idx),
        std::move(post));
    return frame;
}

size_t SyntacticFrameRefiner::update_abstraction(
    const vector<std::uint8_t>& applicable,
    const Cube& cube,
    size_t cube_id)
{
#if VERBOSE_DEBUG_PRINTS
    std::cout << "... inserting cube" << cube_id << " " << cube << std::endl;
#endif

    abstraction_->notify_new_node();
    size_t frame = CubeDatabase::INF_FRAME;
    size_t i = 0;
    auto it = model_->actions.begin();
    for (; i < model_->actions.size() && it->label == SILENT_ACTION;
         ++i, ++it) {
        if (applicable[i]) {
            const auto& action = *it;
            if (applicable[i] & RECHECK_GUARD) {
                if (is_conjunction_violated(action.guard, cube)) {
                    continue;
                }
            }
            auto context_cube = prepare_progression(cube, action.guard);
            for (const auto& out : action.outcomes) {
                frame = std::min(
                    insert_transition(
                        cube_id,
                        cube,
                        context_cube,
                        i,
                        action,
                        out.assignments),
                    frame);
            }
        }
    }

#if VERBOSE_DEBUG_PRINTS
    std::cout << " ... computing edges for CUBE" << cube_id << std::endl;
#endif

    for (; i < model_->actions.size(); ++i, ++it) {
        if (applicable[i]) {
            const auto& action = *it;
            if (applicable[i] & RECHECK_GUARD) {
                if (is_conjunction_violated(action.guard, cube)) {
                    continue;
                }
            }

            auto context_cube = prepare_progression(cube, action.guard);

            if (applicable[i] & RECHECK_POLICY &&
                reasons_->is_blocked(context_cube, action.label)) {
                continue;
            }

            for (const auto& out : action.outcomes) {
                frame = std::min(
                    insert_transition(
                        cube_id,
                        cube,
                        context_cube,
                        i,
                        action,
                        out.assignments),
                    frame);
            }
        }
    }

#if VERBOSE_DEBUG_PRINTS
    std::cout << " ... update existing edges with CUBE" << cube_id << " "
              << cube << std::endl;
#endif

    abstraction_->forall_arcs_where_necessarily(cube, [&](auto& arc) {
        abstraction_->add_successor(arc, cube_id);
    });

#ifndef NDEBUG
    for (size_t node = 0; node < abstraction_->num_nodes(); ++node) {
        auto& arcs = abstraction_->get_node_arcs(node);
        for (const size_t& arc_id : arcs) {
            const auto& arc = abstraction_->get_arc(arc_id);
            vector<size_t> successors = arc.successors;
            std::sort(successors.begin(), successors.end());
            vector<size_t> new_successors =
                get_implied_cubes(abstraction_->get_post_condition(arc));
            std::sort(new_successors.begin(), new_successors.end());
            size_t i = 0, j = 0;
            for (; i < successors.size() && j < new_successors.size();) {
                assert(successors[i] <= new_successors[j]);
                if (successors[i] == new_successors[j]) {
                    ++i;
                    ++j;
                } else if (successors[i] < new_successors[j]) {
                    assert(frames_->is_orphaned(successors[i]));
                    ++i;
                }
            }
            assert(j == new_successors.size());
            for (; i < successors.size(); ++i) {
                assert(frames_->is_orphaned(successors[i]));
            }
        }
    }
#endif

    if (frame == CubeDatabase::INF_FRAME) {
        abstraction_->set_h_value(cube_id, CubeDatabase::INF_FRAME);
        return CubeDatabase::INF_FRAME;
    }
    abstraction_->set_h_value(cube_id, frame + 1);
    return frame + 1;
}

Cube SyntacticFrameRefiner::project_state(
    const flat_state& state,
    const SufficientCondition& cond) const
{
#if VERBOSE_DEBUG_PRINTS
    std::cout << "PROJECT " << print_sequence(state) << std::endl;
    std::cout << "`-> onto: " << print_sequence(cond) << std::endl;
#endif
    Cube cube;
    for (const auto& x : cond) {
        const auto& bound = var_bounds_[x.variable_id].second;
        auto res = cube.emplace(x.variable_id, bound);
        switch (x.type) {
        case VariableCondition::EQUALITY:
            res.first->second.tighten(
                state[x.variable_id],
                state[x.variable_id]);
            break;
        case VariableCondition::LOWER_BOUND:
            res.first->second.tighten(state[x.variable_id], bound.ub);
            break;
        case VariableCondition::UPPER_BOUND:
            res.first->second.tighten(bound.lb, state[x.variable_id]);
            break;
        }
    }
#if VERBOSE_DEBUG_PRINTS
    std::cout << "`-> result: " << cube << std::endl;
#endif
    return cube;
}

SuffCondAlternatives SyntacticFrameRefiner::get_suff_cond_violated(
    const flat_state& state,
    const LinearConstraintConjunction& cond) const
{
    Cube relaxed_state(state);
    return get_suff_cond_violated(state, cond, relaxed_state);
}

SuffCondAlternatives SyntacticFrameRefiner::get_suff_cond_violated(
    const flat_state& state,
    const LinearCondition& cond) const
{
    // get reasons for each individual element of the disjunction
    vector<SuffCondAlternatives> disj_options;
    for (const auto& conj : cond) {
        disj_options.push_back(get_suff_cond_violated(state, conj));
    }
    // do cross product to obtain the reasons why *all* elements of the
    // disjunctions are violated
    return product(disj_options);
}

SuffCondAlternatives SyntacticFrameRefiner::get_suff_cond_violated(
    const flat_state& state,
    const LinearConstraintConjunction& cond,
    Cube& relaxed_state) const
{
    SuffCondAlternatives res;
    for (const auto& constraint : cond) {
        real_t value = 0;
        for (const auto& [var, coef] : constraint) {
            value += static_cast<real_t>(state[var]) * coef;
        }
        SufficientCondition approx;
        switch (constraint.type) {
        case LinearConstraint::EQUAL:
            if (value != constraint.rhs) {
                for (const auto& [var, coef] : constraint) {
                    approx.emplace_back(var, VariableCondition::EQUALITY);
                    relaxed_state[var] = var_bounds_[var];
                }
                res.push_back(std::move(approx));
            }
            break;
        case police::LinearConstraint::GREATER_EQUAL:
            if (value < constraint.rhs) {
                for (const auto& [var, coef] : constraint) {
                    relaxed_state[var] = var_bounds_[var];
                    if (coef < 0.) {
                        approx.emplace_back(
                            var,
                            VariableCondition::LOWER_BOUND);
                    } else {
                        approx.emplace_back(
                            var,
                            VariableCondition::UPPER_BOUND);
                    }
                }
                res.push_back(std::move(approx));
            }
            break;
        case police::LinearConstraint::LESS_EQUAL:
            if (value > constraint.rhs) {
                for (const auto& [var, coef] : constraint) {
                    relaxed_state[var] = var_bounds_[var];
                    if (coef > 0.) {
                        approx.emplace_back(
                            var,
                            VariableCondition::LOWER_BOUND);
                    } else {
                        approx.emplace_back(
                            var,
                            VariableCondition::UPPER_BOUND);
                    }
                }
                res.push_back(std::move(approx));
            }
            break;
        }
    }
    return res;
}

vector<Cube> SyntacticFrameRefiner::get_suff_cond_violated(
    const Cube& state,
    const LinearCondition& cond) const
{
    // get reasons for each individual element of the disjunction
    vector<vector<Cube>> reasons;
    for (const auto& conj : cond) {
        auto r = get_suff_cond_violated(state, conj);
        if (r.empty()) {
            return {};
        }
        reasons.push_back(std::move(r));
    }
    // do cross product to obtain the reasons why *all* elements of the
    // disjunctions are violated
    vector<Cube> result;
    product(reasons, [&result](const vector<const Cube*>& ptrs) {
        Cube c;
        for (const auto* d : ptrs) {
            c &= *d;
        }
        result.push_back(std::move(c));
    });
    return result;
}

vector<Cube> SyntacticFrameRefiner::get_suff_cond_violated(
    const Cube& state,
    const LinearConstraintConjunction& cond) const
{
    vector<Cube> result;
    for (const auto& constraint : cond) {
        real_t lb = 0;
        real_t ub = 0;
        for (const auto& [var, coef] : constraint) {
            if (state.has(var)) {
                const auto& b = state.get(var);
                lb += coef * static_cast<real_t>(coef > 0. ? b.lb : b.ub);
                ub += coef * static_cast<real_t>(coef > 0. ? b.ub : b.lb);
            } else {
                const auto& b = var_bounds_[var].second;
                lb += coef * static_cast<real_t>(coef > 0. ? b.lb : b.ub);
                ub += coef * static_cast<real_t>(coef > 0. ? b.ub : b.lb);
            }
        }
        if (lb > constraint.rhs &&
            (constraint.type == LinearConstraint::EQUAL ||
             constraint.type == LinearConstraint::LESS_EQUAL)) {
            Cube cube;
            for (const auto& [var, coef] : constraint) {
                if (state.has(var)) {
                    cube.set(
                        var,
                        coef > 0. ? Interval(
                                        state.get(var).lb,
                                        var_bounds_[var].second.ub)
                                  : Interval(
                                        var_bounds_[var].second.lb,
                                        state.get(var).ub));
                }
            }
            result.push_back(std::move(cube));
        }
        if (ub < constraint.rhs &&
            (constraint.type == LinearConstraint::EQUAL ||
             constraint.type == LinearConstraint::GREATER_EQUAL)) {
            Cube cube;
            for (const auto& [var, coef] : constraint) {
                if (state.has(var)) {
                    cube.set(
                        var,
                        coef < 0. ? Interval(
                                        state.get(var).lb,
                                        var_bounds_[var].second.ub)
                                  : Interval(
                                        var_bounds_[var].second.lb,
                                        state.get(var).ub));
                }
            }
            result.push_back(std::move(cube));
        }
    }
    return result;
}

SuffCondAlternatives SyntacticFrameRefiner::get_suff_cond_satisfied(
    const flat_state& state,
    const LinearCondition& cond) const
{
    SuffCondAlternatives result;
    // get reasonse for individual elements of the disjunction; final result
    // will be the union of all
    for (const auto& conj : cond) {
        auto varset = get_suff_cond_satisfied(state, conj);
        // varset is empty if conjunction is not satisfied
        if (!varset.empty()) {
            result.push_back(std::move(varset[0]));
        }
    }
    return result;
}

SuffCondAlternatives SyntacticFrameRefiner::get_suff_cond_satisfied(
    const flat_state& state,
    const LinearConstraintConjunction& cond) const
{
    SufficientCondition result;
    for (const auto& constraint : cond) {
        real_t value = 0;
        for (const auto& [var, coef] : constraint) {
            value += static_cast<real_t>(state[var]) * coef;
            result.emplace_back(
                var,
                constraint.type == LinearConstraint::EQUAL
                    ? VariableCondition::EQUALITY
                : (constraint.type == LinearConstraint::LESS_EQUAL &&
                   coef > 0.) ||
                        (constraint.type == LinearConstraint::GREATER_EQUAL &&
                         coef < 0.)
                    ? VariableCondition::UPPER_BOUND
                    : VariableCondition::LOWER_BOUND);
        }
        switch (constraint.type) {
        case LinearConstraint::EQUAL:
            if (value != constraint.rhs) {
                return {};
            }
            break;
        case police::LinearConstraint::GREATER_EQUAL:
            if (value < constraint.rhs) {
                return {};
            }
            break;
        case police::LinearConstraint::LESS_EQUAL:
            if (value > constraint.rhs) {
                return {};
            }
            break;
        }
    }
    make_variable_conditions_unique(model_->variables.size(), result);
    return {std::move(result)};
}

namespace {
Cube remove_trivial_bounds(const Cube& cube, const Cube& bounds)
{
    Cube result;
    for (const auto& [var, vals] : cube) {
        if (vals.has_lb() && vals.lb != bounds[var].second.lb) {
            result.shrink(var, Interval::make_lb(vals.lb));
        }
        if (vals.has_ub() && vals.ub != bounds[var].second.ub) {
            result.shrink(var, Interval::make_ub(vals.ub));
        }
    }
    return result;
}
} // namespace

vector<size_t> SyntacticFrameRefiner::get_satisfied_cubes(
    const Cube& state,
    size_t frame_index) const
{
    vector<size_t> result;
    POLICE_ASSERT(frame_index >= 1u);
    const Cube cube = remove_trivial_bounds(state, var_bounds_);
    frames_->forall_subsumed(cube, {frame_index}, [&](size_t cube) {
        result.push_back(cube);
    });
    return result;
}

vector<size_t> SyntacticFrameRefiner::get_implied_cubes(const Cube& cube) const
{
    const Cube reduced_cube = remove_trivial_bounds(cube, var_bounds_);
    vector<size_t> result;
    frames_->forall_subsumed(reduced_cube, {}, [&](size_t cube_id) {
        result.push_back(cube_id);
    });
    return result;
}

SyntacticFrameRefiner::BoundEstimates SyntacticFrameRefiner::bound_combination(
    const LinearCombination<size_t, real_t>& comb,
    const Cube& cube) const
{
    BoundEstimates est;
    for (const auto& [var, coef] : comb) {
        if (cube.has(var)) {
            const auto& b = cube.get(var);
            est.lb += coef * static_cast<real_t>(coef > 0. ? b.lb : b.ub);
            est.ub += coef * static_cast<real_t>(coef > 0. ? b.ub : b.lb);
        } else {
            const auto& b = var_bounds_[var].second;
            est.lb += coef * static_cast<real_t>(coef > 0. ? b.lb : b.ub);
            est.ub += coef * static_cast<real_t>(coef > 0. ? b.ub : b.lb);
        }
    }
    return est;
}

bool SyntacticFrameRefiner::is_constraint_violated(
    const LinearConstraint& constraint,
    const Cube& cube) const
{
    const auto est = bound_combination(constraint, cube);
    switch (constraint.type) {
    case LinearConstraint::EQUAL:
        return est.lb > constraint.rhs || est.ub < constraint.rhs;
    case LinearConstraint::GREATER_EQUAL: return est.ub < constraint.rhs;
    case LinearConstraint::LESS_EQUAL: return est.lb > constraint.rhs;
    }
    POLICE_UNREACHABLE();
}

bool SyntacticFrameRefiner::is_conjunction_violated(
    const LinearConstraintConjunction& conj,
    const Cube& cube) const
{
    return std::any_of(
        conj.begin(),
        conj.end(),
        [this, &cube](const LinearConstraint& constraint) {
            return is_constraint_violated(constraint, cube);
        });
}

void SyntacticFrameRefiner::drop_trivial_bounds(Cube& cube) const
{
    size_t i = 0;
    for (size_t j = 0; j < cube.size(); ++j) {
        auto var = cube[j].first;
        auto& interval = cube[j].second;
        if (interval != var_bounds_[var].second) {
            if (i != j) {
                cube[i] = std::move(cube[j]);
            }
            ++i;
        }
    }
    cube.erase(cube.begin() + i, cube.end());
}

void SyntacticFrameRefiner::drop_trivial_conditions(
    SufficientCondition& cond,
    const flat_state& state) const
{
    size_t i = 0;
    for (size_t j = 0; j < cond.size(); ++j) {
        auto var_id = cond[j].variable_id;
        if (cond[j].type == VariableCondition::EQUALITY ||
            (cond[j].type == VariableCondition::LOWER_BOUND &&
             state[var_id] > var_bounds_[var_id].second.lb) ||
            (cond[j].type == VariableCondition::UPPER_BOUND &&
             state[var_id] < var_bounds_[var_id].second.ub)) {
            if (i != j) {
                cond[i] = std::move(cond[j]);
            }
            ++i;
        }
    }
    cond.erase(cond.begin() + i, cond.end());
}

void SyntacticFrameRefiner::drop_trivial_conditions(
    SuffCondAlternatives& cond,
    const flat_state& state) const
{
    for (size_t j = 0; j < cond.size(); ++j) {
        drop_trivial_conditions(cond[j], state);
        if (cond[j].empty()) {
            std::swap(cond[j], cond.front());
            cond.resize(1, {});
            return;
        }
    }
}

Cube SyntacticFrameRefiner::prepare_progression(
    const Cube& cube,
    const LinearConstraintConjunction& guard) const
{
    Cube context_cube(var_bounds_);
    for (const auto& [var, vals] : cube) {
        context_cube[var].second.tighten(vals.lb, vals.ub);
    }
    apply_unit_constraints(context_cube, guard);
    propagate_constraints(context_cube, guard);
    return context_cube;
}

std::pair<size_t, bool> SyntacticFrameRefiner::register_reason(Cube reason)
{
    size_t j = 0;
    for (size_t i = 0; i < reason.size(); ++i) {
        const size_t var = reason[i].first;
        auto& vals = reason[i].second;
        if (vals.has_lb() && vals.lb == var_bounds_[var].second.lb) {
            vals.lb = Interval::MIN;
        }
        if (vals.has_ub() && vals.ub == var_bounds_[var].second.ub) {
            vals.ub = Interval::MAX;
        }
        if (vals.has_lb() || vals.has_ub()) {
            if (i != j) {
                reason[j] = std::move(reason[i]);
            }
            ++j;
        }
    }
    reason.erase(reason.begin() + j, reason.end());
    POLICE_DEBUG_MSG(
        "reason after removing trivial bounds: " << reason << "\n");
    return frames_->insert(std::move(reason));
}

} // namespace police::ic3::syntactic

namespace police {
std::ostream& operator<<(
    std::ostream& out,
    const ic3::syntactic::SyntacticFrameRefiner::Statistics& stats)
{
    out << "SynRef total time: " << stats.total_time << "\n";
    out << "SynRef hitting-set time: " << stats.hitting_set_time << "\n";
    out << "SynRef update time: " << stats.update_time << "\n";
    out << "SynRef policy-reasoner time: " << stats.policy_time << "\n";
    out << "SynRef goal reasons: " << stats.goal_reason << "\n";
    out << "SynRef policy-generic reasons: " << stats.no_policy_reason << "\n";
    out << "SynRef policy-specific reasons: " << stats.policy_reason << "\n";
    out << "SynRef total number of policy reasoner calls: "
        << stats.policy_calls_total << "\n";
    out << "SynRef maximal number of policy reasoner calls in one reason "
           "computation: "
        << stats.policy_calls_max << "\n";
    return out;
}

} // namespace police

#undef VERBOSE_DEBUG_PRINTS
