#include "police/verifiers/ic3/syntactic/frames_storage.hpp"
#include "police/verifiers/ic3/syntactic/abstraction.hpp"

#include <algorithm>
#include <fstream>
#include <numeric>

namespace police::ic3::syntactic {

void FramesStorage::json(std::ostream& out, const VariableSpace& variables)
    const
{
    vector<size_t> ordered_cubes(num_cubes(), 0);
    std::iota(ordered_cubes.begin(), ordered_cubes.end(), 0);
    std::sort(
        ordered_cubes.begin(),
        ordered_cubes.end(),
        [&](size_t i, size_t j) {
            const size_t hi = frame_index_->at(i);
            const size_t hj = frame_index_->at(j);
            return std::tie(hi, i) < std::tie(hj, j);
        });
    bool sep = false;
    out << "[\n";
    for (const auto& cube_id : ordered_cubes) {
        const Cube& cube = at(cube_id);
        out << (sep ? ",\n" : "")
            << "  {"
               "\"frame\": "
            << frame_index_->at(cube_id) << ", \"id\": " << cube_id
            << ", \"cube\": " << cube.json(variables) << "}";
        sep = true;
    }
    out << "\n]\n" << std::flush;
}

void FramesStorage::json(std::string_view path, const VariableSpace& variables)
    const
{
    std::ofstream out(path.data());
    json(out, variables);
}

void FramesStorage::dump_frames(
    std::ostream& out,
    const VariableSpace& variables) const
{
    vector<size_t> ordered_cubes(num_cubes(), 0);
    std::iota(ordered_cubes.begin(), ordered_cubes.end(), 0);
    std::sort(
        ordered_cubes.begin(),
        ordered_cubes.end(),
        [&](size_t i, size_t j) {
            const size_t hi = frame_index_->at(i);
            const size_t hj = frame_index_->at(j);
            return std::tie(hi, i) < std::tie(hj, j);
        });
    size_t frame = -1;
    for (const auto& cube_id : ordered_cubes) {
        if (frame_index_->at(cube_id) != frame) {
            frame = frame_index_->at(cube_id);
            if (frame == SyntacticAbstraction::INF_FRAME) {
                out << "----------- frame #inf -----------"
                    << "\n";
            } else {
                out << "----------- frame #" << (frame) << "-----------"
                    << "\n";
            }
        }
        const Cube& cube = at(cube_id);
        out << cube.dump(variables) << "\n";
    }
    out << std::flush;
}

void FramesStorage::json(std::ostream& out, const Model& model) const
{
    vector<size_t> ordered_cubes(num_cubes(), 0);
    std::iota(ordered_cubes.begin(), ordered_cubes.end(), 0);
    std::sort(
        ordered_cubes.begin(),
        ordered_cubes.end(),
        [&](size_t i, size_t j) {
            const size_t hi = frame_index_->at(i);
            const size_t hj = frame_index_->at(j);
            return std::tie(hi, i) < std::tie(hj, j);
        });
    bool sep = false;
    out << "[\n";
    for (const auto& cube_id : ordered_cubes) {
        const Cube& cube = at(cube_id);
        out << (sep ? ",\n" : "")
            << "  {"
               "\"frame\": "
            << frame_index_->at(cube_id) << ", \"id\": " << cube_id
            << ", \"cube\": " << cube.json(model) << "}";
        sep = true;
    }
    out << "\n]\n" << std::flush;
}

void FramesStorage::json(std::string_view path, const Model& model) const
{
    std::ofstream out(path.data());
    json(out, model);
}

} // namespace police::ic3::syntactic
