#pragma once

#include "police/execution_unit.hpp"
#include "police/utils.hpp"

#include "police/storage/path.hpp"
#include "police/verification_property.hpp"
#include "police/verifiers/ic3/concepts.hpp"
#include "police/verifiers/ic3/generic_ic3.hpp"
#include "police/verifiers/ic3/start_generator.hpp"
#include "police/verifiers/ic3/syntactic/abstraction.hpp"
#include "police/verifiers/ic3/syntactic/frame_adder.hpp"
#include "police/verifiers/ic3/syntactic/frame_refinement_wrapper.hpp"
#include "police/verifiers/ic3/syntactic/frame_refiner.hpp"
#include "police/verifiers/ic3/syntactic/frames_storage.hpp"
#include "police/verifiers/ic3/syntactic/frames_storage_utils.hpp"
#include "police/verifiers/ic3/syntactic/policy_reasons.hpp"
#include "police/verifiers/ic3/syntactic/start_avoid_checker.hpp"

#include <iostream>

namespace police::ic3::syntactic {

template <
    search::successor_generator<flat_state> SuccessorGenerator,
    concepts::PathChecker PathChecker>
class IC3Engine
    : public police::ExecutionUnit
    , private ExplicitStateIC3<
          StartGenerator,
          SuccessorGenerator,
          FramesStorageInitializer,
          FrameAdder,
          FramesStorageMemberTest,
          FrameRefinementWrapper,
          PathChecker> {
public:
    IC3Engine(
        const Model* model,
        const VerificationProperty* property,
        StartGenerator start_generator,
        SuccessorGenerator successor_generator,
        SyntacticFrameRefiner frame_refiner,
        PathChecker path_checker,
        StartAvoidChecker start_avoid_checker,
        std::unique_ptr<size_t> cur_frame,
        std::unique_ptr<FramesStorage> frames,
        std::unique_ptr<SyntacticAbstraction> abstraction,
        std::unique_ptr<PolicyReasons> policy_reasons,
        const std::string& store_frames)
        : ExplicitStateIC3<
              StartGenerator,
              SuccessorGenerator,
              FramesStorageInitializer,
              FrameAdder,
              FramesStorageMemberTest,
              FrameRefinementWrapper,
              PathChecker>(
              &model->variables,
              start_generator,
              successor_generator,
              FramesStorageInitializer(frames.get()),
              FrameAdder(
                  frames.get(),
                  abstraction.get(),
                  start_generator,
                  cur_frame.get()),
              FramesStorageMemberTest(frames.get()),
              FrameRefinementWrapper(
                  std::move(frame_refiner),
                  start_generator,
                  frames.get(),
                  cur_frame.get()),
              std::move(path_checker),
              property->avoid,
              false)
        , start_avoid_checker_(std::move(start_avoid_checker))
        , store_frames_(store_frames)
        , reasons_(std::move(policy_reasons))
        , frames_(std::move(frames))
        , abstraction_(std::move(abstraction))
        , cur_frame_(std::move(cur_frame))
        , model_(model)
        , property_(property)
    {
        frames_->set_frame_indexes(&abstraction_->get_h_values());
    }

    std::string_view name() const override { return "syntactic-ic3"; }

    void run() override
    {
        std::cout << "Running SYNIC3..." << std::endl;
        if (start_avoid_checker_()) {
            return;
        }
        result_ = this->operator()();
        statistics_ = this->get_statistics();
        refiner_stats_ = this->frames_refiner().get_statistics();
    }

    void report_result() override
    {
        print_verification_result(std::cout, *model_, result_);
        if (store_frames_.size() > 0u) {
            frames_->json(store_frames_, *model_);
        }
    }

    void report_statistics() override
    {
        std::cout << statistics_;
        std::cout << refiner_stats_;
    }

private:
    StartAvoidChecker start_avoid_checker_;

    std::string store_frames_;

    std::unique_ptr<PolicyReasons> reasons_;
    std::unique_ptr<FramesStorage> frames_;
    std::unique_ptr<SyntacticAbstraction> abstraction_;
    std::unique_ptr<size_t> cur_frame_;

    const Model* model_;
    [[maybe_unused]]
    const VerificationProperty* property_;

    std::optional<Path> result_;
    Statistics statistics_;
    SyntacticFrameRefiner::Statistics refiner_stats_;
};

} // namespace police::ic3::syntactic
