#pragma once

#include "police/verifiers/search/concepts.hpp"
#include "police/storage/segmented_vector.hpp"
#include "police/storage/unordered_set.hpp"
#include "police/storage/vector.hpp"

#include <algorithm>
#include <cassert>
#include <concepts>
#include <cstdint>
#include <iterator>
#include <type_traits>
#include <utility>

namespace police::search {

using feature_t = std::uint32_t;
using feature_vector_t = vector<feature_t>;

template <typename R>
concept feature_range = requires(R r) {
    requires std::forward_iterator<std::decay_t<decltype(r.begin())>>;
    requires std::sentinel_for<
        std::decay_t<decltype(r.end())>,
        std::decay_t<decltype(r.begin())>>;
    { *r.begin() } -> std::same_as<feature_vector_t>;
};

template <typename F, typename StateType>
concept state_feature_map = requires(F f) {
    { f(std::declval<StateType>()) } -> feature_range;
};

template <
    typename StateType,
    search::state_to_id_map<StateType> StateToId,
    state_feature_map<StateType> FeatureMapping>
class NoveltyQueue {
public:
    constexpr explicit NoveltyQueue(
        size_t max_feature_size,
        StateToId state_id_map = StateToId(),
        FeatureMapping features = FeatureMapping())
        : state_id_map_(std::move(state_id_map))
        , get_features_(std::move(features))
        , queues_(max_feature_size + 1)
        , features_(max_feature_size)
        , q_(queues_.size())
    {
    }

    constexpr bool push(const StateType& state)
    {
        size_t novelty = queues_.size() - 1;
        std::ranges::for_each(get_features_(state), [&](auto&& fvec) {
            assert(!fvec.empty());
            if (fvec.size() <= features_.size()) {
                auto is_new = features_[fvec.size() - 1].insert(fvec);
                if (is_new.second) {
                    novelty = fvec.size() <= novelty ? fvec.size() : novelty;
                }
            }
        });
        queues_[novelty].push_back(state_id_map_(state));
        q_ = std::min(q_, novelty);
        return true;
    }

    constexpr NodeId pop()
    {
        assert(!empty());
        auto& queue = queues_[q_];
        NodeId result = queue.back();
        queue.pop_back();
        for (; q_ < queues_.size() && queues_[q_].empty(); ++q_);
        return result;
    }

    [[nodiscard]]
    constexpr bool empty() const
    {
        return q_ == queues_.size();
    }

    void clear()
    {
        const size_t max_size = queues_.size() - 1;
        vector<segmented_vector<NodeId>>(max_size + 1).swap(queues_);
        vector<unordered_set<feature_vector_t>>(max_size).swap(features_);
        q_ = max_size + 1;
    }

private:
    StateToId state_id_map_;
    FeatureMapping get_features_;
    vector<segmented_vector<NodeId>> queues_;
    vector<unordered_set<feature_vector_t>> features_;
    size_t q_;
};

} // namespace police::search
