#include "helpers.hpp"
#include "police/jani/parser/composition.hpp"
#include "police/jani/parser/language.hpp"

#include <catch2/catch.hpp>
#include <initializer_list>
#include <nlohmann/json.hpp>
#include <optional>
#include <string_view>

namespace {
namespace parser = police::jani::parser;
using json = nlohmann::json;
using namespace police::test;

json element(std::string_view automaton)
{
    json obj;
    obj[police::jani::parser::lang::AUTOMATON] = automaton;
    return obj;
}

json element(
    std::string_view automaton,
    std::initializer_list<std::string_view> input_enable)
{
    json obj = element(automaton);
    obj[police::jani::parser::lang::INPUT_ENABLE] = to_list(input_enable);
    return obj;
}

json sync(std::initializer_list<std::string_view> sync, std::string_view result)
{
    json obj;
    obj[police::jani::parser::lang::SYNCHRONISE] = to_list(sync);
    obj[police::jani::parser::lang::RESULT] = result;
    return obj;
}

json composition(std::initializer_list<json> elements)
{
    json obj;
    obj[police::jani::parser::lang::ELEMENTS] = to_list(elements);
    return obj;
}

json composition(
    std::initializer_list<json> elements,
    std::initializer_list<json> syncs)
{
    auto res = composition(elements);
    res[police::jani::parser::lang::SYNCS] = to_list(syncs);
    return res;
}

police::jani::parser::Composition test_composition(const json& js)
{
    auto var = parser::composition_schema()(js, true);
    REQUIRE(var.has_value());
    return var.value();
}

} // namespace

TEST_CASE("Parse composition singleton", "[jani][parser][composition]")
{
    auto c = test_composition(composition({element("a1")}));
    REQUIRE(c.elements.size() == 1);
    REQUIRE(c.elements[0].automaton == "a1");
    REQUIRE(c.elements[0].input_enable.empty());
    REQUIRE(!c.syncs.has_value());
}

TEST_CASE(
    "Parse composition input-enabled singleton",
    "[jani][parser][composition]")
{
    auto c = test_composition(composition({element("a1", {"l1"})}));
    REQUIRE(c.elements.size() == 1);
    REQUIRE(c.elements[0].automaton == "a1");
    REQUIRE(c.elements[0].input_enable.size() == 1);
    REQUIRE(c.elements[0].input_enable[0] == "l1");
    REQUIRE(!c.syncs.has_value());
}

TEST_CASE("Parse composition with syncs", "[jani][parser][composition]")
{
    auto c = test_composition(composition(
        {element("a1"), element("a2")},
        {sync({"k", "k"}, "l"), sync({"l", "l"}, "k")}));
    REQUIRE(c.elements.size() == 2);
    REQUIRE(
        (c.elements[0].automaton == "a1" && c.elements[1].automaton == "a2"));
    REQUIRE(c.syncs.has_value());
    REQUIRE(c.syncs.value().size() == 2);
    REQUIRE(c.syncs.value()[0].synchronise.size() == 2);
    REQUIRE(
        (c.syncs.value()[0].synchronise[0].has_value() &&
         c.syncs.value()[0].synchronise[0].value() == "k"));
    REQUIRE(
        (c.syncs.value()[0].synchronise[1].has_value() &&
         c.syncs.value()[0].synchronise[1].value() == "k"));
    REQUIRE(c.syncs.value()[0].result == "l");
    REQUIRE(c.syncs.value()[1].synchronise.size() == 2);
    REQUIRE(
        (c.syncs.value()[1].synchronise[0].has_value() &&
         c.syncs.value()[1].synchronise[0].value() == "l"));
    REQUIRE(
        (c.syncs.value()[1].synchronise[1].has_value() &&
         c.syncs.value()[1].synchronise[1].value() == "l"));
    REQUIRE(c.syncs.value()[1].result == "k");
}

TEST_CASE("Parse composition with null sync", "[jani][parser][composition]")
{
    auto c = test_composition(composition(
        {element("a1"), element("a2")},
        {sync({"l", "null"}, "l")}));
    REQUIRE(c.elements.size() == 2);
    REQUIRE(
        (c.elements[0].automaton == "a1" && c.elements[1].automaton == "a2"));
    REQUIRE(c.syncs.has_value());
    REQUIRE(c.syncs.value().size() == 1);
    REQUIRE(c.syncs.value()[0].synchronise.size() == 2);
    REQUIRE(
        (c.syncs.value()[0].synchronise[0].has_value() &&
         c.syncs.value()[0].synchronise[0].value() == "l"));
    REQUIRE(!c.syncs.value()[0].synchronise[1].has_value());
    REQUIRE(c.syncs.value()[0].result == "l");
}
