#include "helpers.hpp"
#include "police/jani/parser/constant_declaration.hpp"
#include "police/jani/parser/language.hpp"
#include "police/jani/parser/types.hpp"

#include <catch2/catch.hpp>
#include <nlohmann/json.hpp>
#include <optional>
#include <string_view>

namespace {
namespace parser = police::jani::parser;
using json = nlohmann::json;
using namespace police::test;

json constant(std::string_view var_name, const json& type)
{
    json obj;
    obj[police::jani::parser::lang::NAME] = var_name;
    obj[police::jani::parser::lang::TYPE] = type;
    return obj;
}

json constant(std::string_view var_name, const json& type, const json& value)
{
    json obj = constant(var_name, type);
    obj[police::jani::parser::lang::VALUE] = value;
    return obj;
}

json constant(
    std::string_view var_name,
    const json& type,
    const json& value,
    std::string_view comment)
{
    json obj = constant(var_name, type, value);
    obj[police::jani::parser::lang::COMMENT] = comment;
    return obj;
}

police::jani::parser::ConstantDeclaration test_variable(const json& js)
{
    auto var = parser::constant_declaration_schema()(js, true);
    REQUIRE(var.has_value());
    return var.value();
}

} // namespace

TEST_CASE("Parse int constant", "[jani][parser][constant-declaration]")
{
    auto var = test_variable(constant("x", "int"));
    REQUIRE(var.identifier == "x");
    type_cast<police::jani::parser::IntegerType>(var.type);
    REQUIRE(!var.value.has_value());
}

TEST_CASE(
    "Parse constant with initial value",
    "[jani][parser][constant-declaration]")
{
    auto var = test_variable(constant("x", "int", 100));
    REQUIRE(var.identifier == "x");
    type_cast<police::jani::parser::IntegerType>(var.type);
    REQUIRE((
        var.value.has_value() && var.value.value().is_same(police::Value(100))));
}

TEST_CASE("Parse constant with comment", "[jani][parser][constant-declaration]")
{
    auto var = test_variable(constant("x", "real", 2.0, "ABC"));
    REQUIRE(var.identifier == "x");
    type_cast<police::jani::parser::RealType>(var.type);
    REQUIRE((
        var.value.has_value() && var.value.value().is_same(police::Value(2.0))));
}
