#include "helpers.hpp"
#include "police/jani/parser/model.hpp"

#include <catch2/catch.hpp>
#include <nlohmann/json.hpp>

namespace {
using json = nlohmann::json;
namespace parser = police::jani::parser;
using namespace police::test;

auto test_model(const json& js)
{
    auto model = police::jani::parser::model_schema()(js, true);
    REQUIRE(model.has_value());
    return model.value();
}
} // namespace

TEST_CASE("Parse empty model", "[jani][parser][model]")
{
    auto js = R"({
        "jani-version": 1,
        "name": "model",
        "type": "mdp",
        "automata": [],
        "system": {
            "elements": []
        }
    })"_json;
    auto model = test_model(js);
    REQUIRE(model.name == "model");
    REQUIRE(model.type == parser::ModelType::MDP);
    REQUIRE(model.automata.empty());
    REQUIRE(model.variables.empty());
    REQUIRE((model.system.elements.empty() && !model.system.syncs.has_value()));
    REQUIRE(!model.restrict_initial.has_value());
    REQUIRE(model.properties.empty());
}

TEST_CASE("Parse model", "[jani][parser][model]")
{
    auto js = R"({
        "jani-version": 1,
        "name": "abc",
        "type": "mdp",
        "automata": [
        {
            "name": "a1",
            "locations": [{"name": "l1"}, {"name": "l2"}],
            "initial-locations": ["l1"],
            "edges": []
        },
        {
            "name": "a2",
            "locations": [{"name": "l1"}, {"name": "l2"}],
            "initial-locations": ["l1"],
            "edges": []
        }
        ],
        "actions": [
            {"name": "x"},
            {"name": "y"}
        ],
        "constants": [
            {"name": "const", "type": "int", "value": 100}
        ],
        "variables": [
            {"name": "var", "type": "bool", "initial-value": false}
        ],
        "restrict-initial": { "exp": {
            "op": "=",
            "left": "var",
            "right": true
        }},
        "properties": [
            {
                "name": "goal",
                "expression": {
                    "op": "filter",
                    "fun": "max",
                    "states": {
                        "op": "initial"
                    },
                    "values": "var"
                }
            }
        ],
        "system": {
            "elements": [{"automaton": "a1"}, {"automaton": "a2"}]
        }
    })"_json;
    auto model = test_model(js);
    REQUIRE(model.automata.size() == 2);
    REQUIRE(model.actions.size() == 2);
    REQUIRE(model.constants.size() == 1);
    REQUIRE(model.variables.size() == 1);
    REQUIRE(model.restrict_initial.has_value());
    REQUIRE(model.properties.size() == 1);
    REQUIRE(model.system.elements.size() == 2);
}
