#include "helpers.hpp"
#include "police/jani/parser/language.hpp"
#include "police/jani/parser/types.hpp"
#include "police/jani/parser/variable.hpp"

#include <catch2/catch.hpp>
#include <nlohmann/json.hpp>
#include <optional>
#include <string_view>

namespace {
namespace parser = police::jani::parser;
using json = nlohmann::json;
using namespace police::test;

json variable(std::string_view var_name, const json& type)
{
    json obj;
    obj[police::jani::parser::lang::NAME] = var_name;
    obj[police::jani::parser::lang::TYPE] = type;
    return obj;
}

json variable(
    std::string_view var_name,
    const json& type,
    const json& initial_value)
{
    json obj = variable(var_name, type);
    obj[police::jani::parser::lang::INITIAL_VALUE] = initial_value;
    return obj;
}

json variable(
    std::string_view var_name,
    const json& type,
    const json& initial_value,
    std::string_view comment)
{
    json obj = variable(var_name, type, initial_value);
    obj[police::jani::parser::lang::COMMENT] = comment;
    return obj;
}

police::jani::parser::VariableDeclaration test_variable(const json& js)
{
    auto var = parser::variable_schema()(js, true);
    REQUIRE(var.has_value());
    return var.value();
}

} // namespace

TEST_CASE("Parse int variable", "[jani][parser][variable-declaration]")
{
    auto var = test_variable(variable("x", "int"));
    REQUIRE(var.identifier == "x");
    type_cast<police::jani::parser::IntegerType>(var.type);
    REQUIRE(!var.initial_value.has_value());
}

TEST_CASE(
    "Parse variable with initial value",
    "[jani][parser][variable-declaration]")
{
    auto var = test_variable(variable("x", "int", 100));
    REQUIRE(var.identifier == "x");
    type_cast<police::jani::parser::IntegerType>(var.type);
    REQUIRE(var.initial_value.has_value());
    require_is_constant(var.initial_value.value(), 100);
}

TEST_CASE("Parse variable with comment", "[jani][parser][variable-declaration]")
{
    auto var = test_variable(variable("x", "real", 2.0, "ABC"));
    REQUIRE(var.identifier == "x");
    type_cast<police::jani::parser::RealType>(var.type);
    REQUIRE(var.initial_value.has_value());
    require_is_constant(var.initial_value.value(), 2.0);
}
