#include "simple_counter.hpp"
#include "police/action.hpp"
#include "police/base_types.hpp"
#include "police/storage/variable_space.hpp"

#include <cassert>

namespace {
using namespace police;

LinearConstraint
create_constraint(police::size_t var, int val, LinearConstraint::Type type)
{
    LinearConstraint res(type);
    res.insert(var, 1.);
    res.rhs = val;
    return res;
}

LinearExpression create_const_expression(int val)
{
    LinearExpression res;
    res.bias = val;
    return res;
}

LinearExpression create_expression(police::size_t var, int addand)
{
    auto res = create_const_expression(addand);
    res.insert(var, 1.);
    return res;
}

} // namespace

Model simple_counter_model(int count_to, bool)
{
    assert(count_to > 0);
    VariableSpace variables;
    const auto counter =
        variables.add_variable("counter", BoundedIntType(0, count_to));
    vector<Action> actions;
    {
        LinearConstraintConjunction guard;
        guard &= create_constraint(
            counter,
            count_to - 1,
            LinearConstraint::Type::LESS_EQUAL);
        vector<Assignment> effs;
        effs.emplace_back(counter, create_expression(counter, 1));
        actions.push_back(
            Action(0, std::move(guard), {Outcome(std::move(effs))}));
    }
    {
        LinearConstraintConjunction guard;
        guard &= create_constraint(
            counter,
            1,
            LinearConstraint::Type::GREATER_EQUAL);
        vector<Assignment> effs;
        effs.emplace_back(counter, create_expression(counter, -1));
        actions.push_back(
            Action(1, std::move(guard), {Outcome(std::move(effs))}));
    }
    return Model(std::move(variables), std::move(actions), {});
}

police::LinearCondition simple_counter_goal(int count_to)
{
    police::LinearCondition res;
    police::LinearConstraint con(police::LinearConstraint::EQUAL);
    con.insert(0, 1.);
    con.rhs = count_to;
    res |= con;
    return res;
}
