#include "police/storage/dynamic_bit_field.hpp"

#include <algorithm>
#include <catch2/catch.hpp>

namespace {
using namespace police;
}

TEST_CASE("Empty bit field construction", "[storage][dynamic_bitfield]")
{
    dynamic_bit_field<> bf{};
    REQUIRE(bf.size() == 0u);
    REQUIRE(bf.capacity() == 0u);
}

TEST_CASE("Bit field default construction", "[storage][dynamic_bitfield]")
{
    SECTION("Check true")
    {
        dynamic_bit_field<> bf(100, true);
        REQUIRE(bf.size() == 100u);
        REQUIRE(std::all_of(bf.begin(), bf.end(), [](auto&& val) -> bool {
            return val;
        }));
    }
    SECTION("Check false")
    {
        dynamic_bit_field<> bf(100, false);
        REQUIRE(bf.size() == 100u);
        REQUIRE(std::all_of(bf.begin(), bf.end(), [](auto&& val) -> bool {
            return !val;
        }));
    }
}

TEST_CASE("Bit field manipulation", "[storage][dynamic_bitfield]")
{
    dynamic_bit_field<> bf(100, false);
    REQUIRE(bf.size() == 100u);
    for (auto i = 0u; i < bf.size(); ++i) {
        bf[i] = i % 2;
    }
    for (auto i = 0u; i < bf.size(); ++i) {
        CHECK((i % 2 == 1) == bf[i]);
    }
}

TEST_CASE("Bit field resize grow", "[storage][dynamic_bitfield]")
{
    SECTION("Within bucket")
    {
        dynamic_bit_field<> bf(32, false);
        REQUIRE(bf.size() == 32u);
        CHECK(std::all_of(bf.begin(), bf.end(), [](auto x) -> bool {
            return !x;
        }));
        bf.resize(64, true);
        REQUIRE(bf.size() == 64u);
        CHECK(std::all_of(bf.begin(), bf.begin() + 32u, [](auto x) -> bool {
            return !x;
        }));
        CHECK(std::all_of(bf.begin() + 32u, bf.end(), [](auto x) -> bool {
            return x;
        }));
    }
    SECTION("Bucket plus one")
    {
        dynamic_bit_field<> bf(32, false);
        REQUIRE(bf.size() == 32u);
        CHECK(std::all_of(bf.begin(), bf.end(), [](auto x) -> bool {
            return !x;
        }));
        bf.resize(65, true);
        REQUIRE(bf.size() == 65u);
        CHECK(std::all_of(bf.begin(), bf.begin() + 32u, [](auto x) -> bool {
            return !x;
        }));
        CHECK(std::all_of(bf.begin() + 32u, bf.end(), [](auto x) -> bool {
            return x;
        }));
    }
    SECTION("Multiple buckets")
    {
        dynamic_bit_field<> bf(100, false);
        REQUIRE(bf.size() == 100u);
        for (auto i = 0u; i < bf.size(); ++i) {
            bf[i] = i % 2;
        }
        bf.resize(200, true);
        REQUIRE(bf.size() == 200u);
        for (auto i = 0u; i < 100u; ++i) {
            CHECK((i % 2 == 1) == bf[i]);
        }
        CHECK(std::all_of(bf.begin() + 100, bf.end(), [](auto x) -> bool {
            return x;
        }));
    }
}

TEST_CASE("Bit field resize shrink", "[storage][dynamic_bitfield]")
{
    SECTION("Within bucket")
    {
        dynamic_bit_field<> bf(32, false);
        REQUIRE(bf.size() == 32u);
        CHECK(std::all_of(bf.begin(), bf.end(), [](auto x) -> bool {
            return !x;
        }));
        bf.resize(16, true);
        REQUIRE(bf.size() == 16u);
        CHECK(std::all_of(bf.begin(), bf.end(), [](auto x) -> bool {
            return !x;
        }));
    }
    SECTION("Bucket minus one")
    {
        dynamic_bit_field<> bf(65, true);
        REQUIRE(bf.size() == 65u);
        CHECK(std::all_of(bf.begin(), bf.end(), [](auto x) -> bool {
            return x;
        }));
        bf.resize(64, false);
        REQUIRE(bf.size() == 64u);
        CHECK(std::all_of(bf.begin(), bf.end(), [](auto x) -> bool {
            return x;
        }));
    }
    SECTION("Multiple buckets")
    {
        dynamic_bit_field<> bf(200, true);
        REQUIRE(bf.size() == 200u);
        CHECK(std::all_of(bf.begin(), bf.end(), [](auto x) -> bool {
            return x;
        }));
        bf.resize(100, true);
        REQUIRE(bf.size() == 100u);
        CHECK(std::all_of(bf.begin(), bf.end(), [](auto x) -> bool {
            return x;
        }));
    }
}

TEST_CASE("Bit field push_back and pop_back", "[storage][dynamic_bitfield]")
{
    dynamic_bit_field<> bf{};
    for (auto i = 0u; i < 200u; ++i) {
        bf.push_back(i % 2);
    }
    REQUIRE(bf.size() == 200u);
    for (auto i = 0u; i < 200u; ++i) {
        CHECK(bf[i] == (i % 2 == 1));
    }
    for (auto i = 100u; i > 0; --i) {
        bf.pop_back();
    }
    REQUIRE(bf.size() == 100u);
    for (auto i = 0u; i < 100u; ++i) {
        CHECK(bf[i] == (i % 2 == 1));
    }
}

TEST_CASE("Bit field comparison", "[storage][dynamic_bitfield]")
{
    SECTION("Equal empty bfs")
    {
        dynamic_bit_field<> bf0{};
        dynamic_bit_field<> bf1{};
        REQUIRE(bf0 == bf1);
    }
    SECTION("Equal empty bfs after reserve")
    {
        dynamic_bit_field<> bf0{};
        bf0.reserve(100);
        dynamic_bit_field<> bf1{};
        REQUIRE(bf0 == bf1);
    }
    SECTION("Equal true bf")
    {
        dynamic_bit_field<> bf0(100, false);
        dynamic_bit_field<> bf1(100, false);
        REQUIRE(bf0 == bf1);
    }
    SECTION("Not equal bit fields, same size")
    {
        dynamic_bit_field<> bf0(100, true);
        dynamic_bit_field<> bf1(100, false);
        REQUIRE(bf0 != bf1);
    }
    SECTION("Not equal bit fields, same size 2")
    {
        dynamic_bit_field<> bf0(100, true);
        dynamic_bit_field<> bf1(100, true);
        bf1[89] = false;
        REQUIRE(bf0 != bf1);
    }
    SECTION("Not equal bit fields, different size")
    {
        dynamic_bit_field<> bf0(100, true);
        dynamic_bit_field<> bf1(99, true);
        REQUIRE(bf0 != bf1);
    }
    SECTION("Equal after modification")
    {
        dynamic_bit_field<> bf0{};
        dynamic_bit_field<> bf1{};
        for (auto i = 0u; i < 200u; ++i) {
            bf0.push_back(i % 2);
        }
        for (auto i = 0u; i < 100u; ++i) {
            bf1.push_back(i % 2);
        }
        bf0.resize(100u);
        REQUIRE(bf0 == bf1);
    }
}
