#pragma once

#include "police/base_types.hpp"
#include "police/global_arguments.hpp"
#include "police/option_parser.hpp"
#include "police/storage/unordered_map.hpp"
#include "police/storage/vector.hpp"

#include <functional>
#include <initializer_list>
#include <optional>
#include <string>
#include <string_view>

namespace police {

class CommandLineParser {
public:
    using OptionId = size_t;

    CommandLineParser() = default;

    OptionId add_action(
        std::string_view name,
        std::function<void(GlobalArguments&)> action,
        std::string_view help = "",
        std::initializer_list<std::string_view> dependencies = {},
        bool mandatory = false,
        std::optional<std::function<void(GlobalArguments&)>> fallback =
            std::nullopt);

    OptionId add_flag(
        std::string_view name,
        bool GlobalArguments::* dest,
        std::string_view help = "",
        std::initializer_list<std::string_view> dependencies = {},
        std::optional<std::function<void(GlobalArguments&)>> fallback =
            std::nullopt);

    OptionId add_raw_argument(
        std::string_view name,
        std::function<void(GlobalArguments&, std::string_view)> store_arg,
        std::string_view help = "",
        std::initializer_list<std::string_view> dependencies = {},
        bool mandatory = false,
        std::optional<std::function<void(GlobalArguments&)>> fallback =
            std::nullopt);

    template <typename T>
    OptionId add_argument(
        std::string_view name,
        std::function<void(GlobalArguments&, T)> store_arg,
        std::string_view help = "",
        std::initializer_list<std::string_view> dependencies = {},
        bool mandatory = false,
        std::optional<std::function<void(GlobalArguments&)>> fallback =
            std::nullopt)
    {
        auto parse = [store_arg](GlobalArguments& store, std::string_view arg) {
            T value = OptionParser::parse<T>(store, arg);
            store_arg(store, std::move(value));
        };
        return add_raw_argument(
            name,
            std::move(parse),
            help,
            std::move(dependencies),
            mandatory,
            std::move(fallback));
    }

    template <typename T>
    OptionId add_argument(
        std::string_view name,
        T GlobalArguments::* dest,
        std::string_view help = "",
        std::initializer_list<std::string_view> dependencies = {},
        bool mandatory = false,
        std::optional<T> default_value = std::nullopt)
    {
        auto parse = [dest](GlobalArguments& store, std::string_view arg) {
            T value = OptionParser::parse<T>(store, arg);
            (store.*dest) = std::move(value);
        };
        if (default_value == std::nullopt) {
            return add_raw_argument(
                name,
                std::move(parse),
                help,
                std::move(dependencies),
                mandatory);
        } else {
            auto fallback = [&](GlobalArguments& store) {
                (store.*dest) = default_value.value();
            };
            return add_raw_argument(
                name,
                std::move(parse),
                help,
                std::move(dependencies),
                mandatory,
                std::move(fallback));
        }
    }

    void create_dependency_group(
        std::string_view name,
        std::initializer_list<std::string_view> members);

    void ensure_oneof(std::initializer_list<OptionId> options);

    [[nodiscard]]
    GlobalArguments parse(const vector<std::string_view>& args);

    [[nodiscard]]
    GlobalArguments parse(int argc, const char** argv);

    [[nodiscard]]
    bool has_argument(const std::string& name) const;

private:
    unordered_map<std::string, OptionId> options_;
    unordered_map<std::string, size_t> group_to_id_;
    vector<vector<std::string>> dependency_groups_;
    vector<std::string> names_;
    vector<std::function<void(GlobalArguments&, std::string_view)>> parsers_;
    vector<std::optional<std::function<void(GlobalArguments&)>>> fallback_;
    vector<bool> needs_arg_;
    vector<std::string> help_;
    vector<vector<std::string>> dependencies_;
    vector<vector<OptionId>> mandatory_;
};

} // namespace police
