#include "police/expressions/binary_function_call.hpp"
#include "police/expressions/expression.hpp"
#include "police/expressions/expression_transformer.hpp"
#include "police/expressions/expression_visitor.hpp"
#include "police/utils/hash.hpp"

namespace police::expressions {

BinaryFunctionCall::BinaryFunctionCall(
    Function function,
    Expression left,
    Expression right)
    : function(std::move(function))
    , left(std::move(left))
    , right(std::move(right))
{
}

bool BinaryFunctionCall::operator==(const BinaryFunctionCall& other) const
{
    return function == other.function && left.is_same(other.left) &&
           right.is_same(other.right);
}

void BinaryFunctionCall::dump(std::ostream& out) const
{
    if (function == Function::POW) {
        out << "pow";
    } else {
        out << "log";
    }
    out << "(" << left << ", " << right << ")";
}

std::size_t BinaryFunctionCall::hash() const
{
    return hash_combine(
        typeid(BinaryFunctionCall).hash_code(),
        get_hash(static_cast<int>(function)),
        get_hash(left),
        get_hash(right));
}

__POLICE_IMPLEMENT_EXPR_VISITORS(BinaryFunctionCall)

BinaryFunctionCallGeneric::BinaryFunctionCallGeneric(
    Function function,
    Expression left,
    Expression right)
    : function(std::move(function))
    , left(std::move(left))
    , right(std::move(right))
{
}

bool BinaryFunctionCallGeneric::operator==(
    const BinaryFunctionCallGeneric& other) const
{
    return function == other.function && left.is_same(other.left) &&
           right.is_same(other.right);
}

void BinaryFunctionCallGeneric::dump(std::ostream& out) const
{
    if (function == Function::MIN) {
        out << "min";
    } else {
        out << "max";
    }
    out << "(" << left << ", " << right << ")";
}

std::size_t BinaryFunctionCallGeneric::hash() const
{
    return hash_combine(
        typeid(BinaryFunctionCallGeneric).hash_code(),
        get_hash(static_cast<int>(function)),
        get_hash(left),
        get_hash(right));
}

__POLICE_IMPLEMENT_EXPR_VISITORS(BinaryFunctionCallGeneric)

} // namespace police::expressions
