#include "police/expressions/expression_transformer.hpp"
#include "police/expressions/expression.hpp"
#include "police/expressions/expressions.hpp"
#include <algorithm>

namespace police::expressions {

namespace {
template <typename Collection>
void apply_to_collection(
    ExpressionTransformer& transformer,
    Collection& children)
{
    std::for_each(children.begin(), children.end(), [&](Expression& child) {
        child.transform(transformer);
    });
    children.erase(
        std::stable_partition(
            children.begin(),
            children.end(),
            [](const Expression& child) { return child.base() != nullptr; }),
        children.end());
}

template <typename... Ptr>
void apply_to_all(ExpressionTransformer& transformer, Ptr&&... ptrs)
{
    (..., ptrs.transform(transformer));
}
} // namespace

void ExpressionTransformer::visit(Expression&, BinaryFunctionCall& expr)
{
    apply_to_all(*this, expr.left, expr.right);
}

void ExpressionTransformer::visit(Expression&, BinaryFunctionCallGeneric& expr)
{
    apply_to_all(*this, expr.left, expr.right);
}

void ExpressionTransformer::visit(Expression&, Conjunction& expr)
{
    apply_to_collection(*this, expr.children);
}

void ExpressionTransformer::visit(Expression&, Disjunction& expr)
{
    apply_to_collection(*this, expr.children);
}

void ExpressionTransformer::visit(Expression&, Comparison& expr)
{
    apply_to_all(*this, expr.left, expr.right);
}

void ExpressionTransformer::visit(Expression&, Constant&)
{
}

void ExpressionTransformer::visit(Expression&, Derivative&)
{
}

void ExpressionTransformer::visit(Expression&, FunctionCall& expr)
{
    apply_to_all(*this, expr.expr);
}

void ExpressionTransformer::visit(Expression&, IdentifierReference&)
{
}

void ExpressionTransformer::visit(Expression&, IfThenElse& expr)
{
    apply_to_all(*this, expr.condition, expr.consequence, expr.alternative);
}

void ExpressionTransformer::visit(Expression&, Negation& expr)
{
    apply_to_all(*this, expr.expr);
}

void ExpressionTransformer::visit(Expression&, NumericOperation& expr)
{
    apply_to_all(*this, expr.left, expr.right);
}

void ExpressionTransformer::visit(Expression&, Variable&)
{
}

} // namespace police::expressions
