#include "police/expressions/until_property.hpp"
#include "police/expressions/expression_transformer.hpp"
#include "police/expressions/expression_visitor.hpp"
#include <algorithm>

namespace police::expressions {

void PropertyInterval::accept(ExpressionVisitor& visitor) const
{
    if (lower.has_value()) lower.value().accept(visitor);
    if (upper.has_value()) upper.value().accept(visitor);
}

void PropertyInterval::transform(ExpressionTransformer& transformer)
{
    if (lower.has_value()) lower.value().transform(transformer);
    if (upper.has_value()) upper.value().transform(transformer);
}

void RewardBound::accept(ExpressionVisitor& visitor) const
{
    expr.accept(visitor);
    bounds.accept(visitor);
}

void RewardBound::transform(ExpressionTransformer& transformer)
{
    expr.transform(transformer);
    bounds.transform(transformer);
}

UntilProperty::UntilProperty(
    Operator op,
    PropertyPtr left,
    PropertyPtr right,
    PropertyInterval step_bounds,
    PropertyInterval time_bounds,
    std::vector<RewardBound> bounds)
    : op(std::move(op))
    , left(std::move(left))
    , right(std::move(right))
    , step_bounds(std::move(step_bounds))
    , time_bounds(std::move(time_bounds))
    , bounds(std::move(bounds))
{
}

void UntilProperty::accept(ExpressionVisitor& visitor) const
{
    left->accept(visitor);
    right->accept(visitor);
    step_bounds.accept(visitor);
    time_bounds.accept(visitor);
    std::for_each(bounds.begin(), bounds.end(), [&](auto&& bound) {
        bound.accept(visitor);
    });
}

void UntilProperty::transform(ExpressionTransformer& transformer)
{
    left->transform(transformer);
    right->transform(transformer);
    step_bounds.transform(transformer);
    time_bounds.transform(transformer);
    std::for_each(bounds.begin(), bounds.end(), [&](auto&& bound) {
        bound.transform(transformer);
    });
}

} // namespace police::expressions
