#pragma once

#include "police/jani/parser/automaton.hpp"
#include "police/jani/parser/composition.hpp"
#include "police/jani/parser/constant_declaration.hpp"
#include "police/jani/parser/expression.hpp"
#include "police/jani/parser/property.hpp"
#include "police/jani/parser/schema_factory.hpp"
#include "police/jani/parser/variable.hpp"

namespace police::jani::parser {

struct Metadata {
    std::string version;
    std::string author;
    std::string description;
    std::string doi;
    std::string url;
};

enum class ModelType {
    LTS,
    DTMC,
    CTMC,
    MDP,
    CTMDP,
    MA,
    TA,
    PTA,
    STA,
    HA,
    PHA,
    SHA
};

enum class ModelFeatures {
    ARRAYS,
    DATATYPES,
    DERIVED_OPERATORS,
    EDGE_PRIORITIES,
    FUNCTIONS,
    HYPERBOLIC_FUNCTIONS,
    NAMED_EXPRESSIONS,
    NONDET_SELECTION,
    STATE_EXIT_REWARDS,
    TRADEOFF_PROPERTIES,
    TRIGONOMETRIC_FUNCTIONS
};

struct Model {
    int jani_version;
    identifier_name_t name;
    Metadata metadata;
    ModelType type;
    std::vector<ModelFeatures> features;
    std::vector<identifier_name_t> actions;
    std::vector<ConstantDeclaration> constants;
    std::vector<VariableDeclaration> variables;
    std::optional<Expression> restrict_initial = std::nullopt;
    std::vector<Property> properties;
    std::vector<Automaton> automata;
    Composition system;
};

JaniSchema<Model> model_schema();

template <
    typename F,
    typename Model_cv,
    std::enable_if_t<
        std::is_same_v<std::remove_cvref_t<Model_cv>, Model>,
        int> = 0>
void apply_to_all_expressions(Model_cv&& model, F fn = F())
{
    for (auto& constant : model.constants) {
        if (constant.value.has_value()) {
            fn(constant.value.value());
        }
    }
    for (auto& var : model.variables) {
        apply_to_all_expressions(var, fn);
    }
    if (model.restrict_initial.has_value()) {
        fn(model.restrict_initial.value());
    }
    for (auto& automaton : model.automata) {
        apply_to_all_expressions(automaton, fn);
    }
    std::for_each(
        model.properties.begin(),
        model.properties.end(),
        [&](auto&& property) { fn(*property.expr); });
}

void visit_all(const Model& model, expressions::ExpressionVisitor& visitor);

void transform_all(
    Model& model,
    expressions::ExpressionTransformer& transformer);

} // namespace police::jani::parser
