#include "police/jani/parser/property.hpp"
#include "police/expressions/constants.hpp"
#include "police/expressions/properties.hpp"
#include "police/expressions/property.hpp"
#include "police/expressions/state_property.hpp"
#include "police/expressions/until_property.hpp"
#include "police/jani/parser/comment.hpp"
#include "police/jani/parser/expression.hpp"
#include "police/jani/parser/language.hpp"
#include "police/jani/parser/schema_factory.hpp"
#include "police/macros.hpp"

namespace police::jani::parser {

namespace {

using expressions::PropertyPtr;

template <typename T>
using MakeExpr = factories::MakeShared<expressions::Property, T>;

struct MakeRewardProperty {
    PropertyPtr operator()(
        expressions::RewardProperty::Operator op,
        Expression expr,
        std::optional<expressions::RewardAccumulation> accumulate,
        PropertyPtr reach,
        std::optional<Expression> step_instant,
        std::optional<Expression> time_instant,
        std::vector<expressions::RewardInstant> reward_instants) const
    {
        return MakeExpr<expressions::RewardProperty>()(
            op,
            std::move(expr),
            accumulate.has_value() ? accumulate.value()
                                   : expressions::RewardAccumulation::NONE,
            std::move(reach),
            std::move(step_instant),
            std::move(time_instant),
            std::move(reward_instants));
    }
};

struct MakeLongRunProperty {
    PropertyPtr operator()(
        expressions::LongRunProperty::Operator op,
        PropertyPtr expr,
        std::optional<expressions::RewardAccumulation> accumulate) const
    {
        return MakeExpr<expressions::LongRunProperty>()(
            op,
            std::move(expr),
            accumulate.has_value() ? accumulate.value()
                                   : expressions::RewardAccumulation::NONE);
    }
};

struct MakeUntilProperty {
    enum class Operand { FINALLY, GLOBALLY };

    template <typename... Args>
    PropertyPtr operator()(Operand op, PropertyPtr expr, Args&&... args) const
    {
        switch (op) {
        case Operand::FINALLY:
            return MakeExpr<expressions::UntilProperty>()(
                expressions::UntilProperty::Operator::UNTIL,
                std::make_shared<expressions::StateProperty>(
                    expressions::MakeConstant()(true)),
                std::move(expr),
                std::forward<Args>(args)...);
        case Operand::GLOBALLY:
            return MakeExpr<expressions::UntilProperty>()(
                expressions::UntilProperty::Operator::WEAK_UNTIL,
                std::move(expr),
                std::make_shared<expressions::StateProperty>(
                    expressions::MakeConstant()(false)),
                std::forward<Args>(args)...);
        }
        POLICE_UNREACHABLE();
    }
};

} // namespace

JaniSchema<PropertyPtr> property_expr_schema()
{
    static JaniSchema<PropertyPtr> schema{};
    static bool init = false;
    if (!init) {
        auto property_interval = make_dictionary(
            factories::Construct<expressions::PropertyInterval>(),
            {},
            JaniDictArgument<std::optional<Expression>>(
                lang::LOWER,
                expression_schema(),
                true),
            JaniDictArgument<std::optional<Expression>>(
                lang::UPPER,
                expression_schema(),
                true),
            JaniDictArgument(
                lang::LOWER_EXCLUSIVE,
                JaniBool<factories::Construct<bool>>()),
            JaniDictArgument(
                lang::UPPER_EXCLUSIVE,
                JaniBool<factories::Construct<bool>>()));

        auto reward_accu = JaniEnum<
            factories::Construct<expressions::RewardAccumulation>,
            lang::STEPS,
            lang::TIME>();

        auto reward_accu_o = JaniEnum<
            factories::OptionalConstruct<expressions::RewardAccumulation>,
            lang::STEPS,
            lang::TIME>();

        auto filter_expr = make_dictionary(
            MakeExpr<expressions::FilterProperty>(),
            {JaniDictElement(
                lang::OP,
                JaniEnum<factories::Discard, lang::FILTER>())},
            JaniDictArgument(
                lang::FUN,
                JaniEnum<
                    factories::Construct<expressions::FilterProperty::Function>,
                    lang::MIN,
                    lang::MAX,
                    lang::SUM,
                    lang::AVG,
                    lang::COUNT,
                    lang::FORALL,
                    lang::EXISTS,
                    lang::ARGMIN,
                    lang::ARGMAX,
                    lang::VALUES>()),
            JaniDictArgument(lang::VALUES, schema),
            JaniDictArgument(lang::STATES, schema));

        auto prob_expr = make_dictionary(
            MakeExpr<expressions::ProbabilityProperty>(),
            {},
            JaniDictArgument(
                lang::OP,
                JaniEnum<
                    factories::Construct<
                        expressions::ProbabilityProperty::Operator>,
                    lang::PMIN,
                    lang::PMAX>()),
            JaniDictArgument(lang::EXP, schema));

        auto path_expr = make_dictionary(
            MakeExpr<expressions::PathProperty>(),
            {},
            JaniDictArgument(
                lang::OP,
                JaniEnum<
                    factories::Construct<expressions::PathProperty::Operator>,
                    lang::FORALL,
                    lang::EXISTS>()),
            JaniDictArgument(lang::EXP, schema));

        auto exp_expr = make_dictionary(
            MakeRewardProperty(),
            {},
            JaniDictArgument(
                lang::OP,
                JaniEnum<
                    factories::Construct<expressions::RewardProperty::Operator>,
                    lang::EMIN,
                    lang::EMAX>()),
            JaniDictArgument(lang::EXP, expression_schema()),
            JaniDictArgument(lang::ACCUMULATE, reward_accu_o, true),
            JaniDictArgument(lang::REACH, schema, true),
            JaniDictArgument<std::optional<Expression>>(
                lang::STEP_INSTANT,
                expression_schema(),
                true),
            JaniDictArgument<std::optional<Expression>>(
                lang::TIME_INSTANT,
                expression_schema(),
                true),
            JaniDictArgument(
                lang::REWARD_INSTANTS,
                JaniList(make_dictionary(
                    factories::Construct<expressions::RewardInstant>(),
                    {},
                    JaniDictArgument(lang::EXP, expression_schema()),
                    JaniDictArgument(lang::ACCUMULATE, reward_accu),
                    JaniDictArgument(lang::INSTANT, expression_schema()))),
                true));

        auto longavg_expr = make_dictionary(
            MakeLongRunProperty(),
            {},
            JaniDictArgument(
                lang::OP,
                JaniEnum<
                    factories::Construct<
                        expressions::LongRunProperty::Operator>,
                    lang::SMIN,
                    lang::SMAX>()),
            JaniDictArgument(lang::EXP, schema),
            JaniDictArgument(lang::ACCUMULATE, reward_accu_o, true));

        auto until_expr = make_dictionary(
            MakeExpr<expressions::UntilProperty>(),
            {},
            JaniDictArgument(
                lang::OP,
                JaniEnum<
                    factories::Construct<expressions::UntilProperty::Operator>,
                    lang::UNTIL,
                    lang::WEAK_UNTIL>()),
            JaniDictArgument(lang::LEFT, schema),
            JaniDictArgument(lang::RIGHT, schema),
            JaniDictArgument(lang::STEP_BOUNDS, property_interval, true),
            JaniDictArgument(lang::TIME_BOUNDS, property_interval, true),
            JaniDictArgument(
                lang::REWARD_BOUNDS,
                JaniList(make_dictionary(
                    factories::Construct<expressions::RewardBound>(),
                    {},
                    JaniDictArgument(lang::EXP, expression_schema()),
                    JaniDictArgument(lang::ACCUMULATE, reward_accu),
                    JaniDictArgument(lang::BOUNDS, property_interval))),
                true));

        auto derived_until = make_dictionary(
            MakeUntilProperty(),
            {},
            JaniDictArgument(
                lang::OP,
                JaniEnum<
                    factories::Construct<MakeUntilProperty::Operand>,
                    lang::FINALLY,
                    lang::GLOBALLY>()),
            JaniDictArgument(lang::EXP, schema),
            JaniDictArgument(lang::STEP_BOUNDS, property_interval, true),
            JaniDictArgument(lang::TIME_BOUNDS, property_interval, true),
            JaniDictArgument(
                lang::REWARD_BOUNDS,
                JaniList(make_dictionary(
                    factories::Construct<expressions::RewardBound>(),
                    {},
                    JaniDictArgument(lang::EXP, expression_schema()),
                    JaniDictArgument(lang::ACCUMULATE, reward_accu),
                    JaniDictArgument(lang::BOUNDS, property_interval))),
                true));

        auto pred_expr = make_dictionary(
            MakeExpr<expressions::StatePredicateProperty>(),
            {},
            JaniDictArgument(
                lang::OP,
                JaniEnum<
                    factories::Construct<
                        expressions::StatePredicateProperty::Predicate>,
                    lang::INITIAL,
                    lang::DEADLOCK,
                    lang::TIMELOCK>()));

        schema = JaniMultiSchemata(
            make_cast(
                MakeExpr<expressions::StateProperty>(),
                expression_schema()),
            std::move(filter_expr),
            std::move(prob_expr),
            std::move(path_expr),
            std::move(exp_expr),
            std::move(longavg_expr),
            std::move(until_expr),
            std::move(derived_until),
            std::move(pred_expr));

        init = true;
    }
    return schema;
}

JaniSchema<Property> property_schema()
{
    static JaniSchema<Property> schema{};
    static bool init = false;
    if (!init) {
        schema = (make_dictionary(
            factories::Construct<Property>(),
            {comment_element},
            JaniDictArgument(
                lang::NAME,
                JaniString<factories::Construct<identifier_name_t>>()),
            JaniDictArgument(lang::EXPRESSION, property_expr_schema())));
        init = true;
    }
    return schema;
}

} // namespace police::jani::parser
