#pragma once

#include "police/expressions/expression.hpp"
#include "police/linear_constraint.hpp"
#include "police/storage/vector.hpp"

#include <algorithm>

namespace police {

struct LinearConstraintConjunction : public vector<LinearConstraint> {
    LinearConstraintConjunction& operator&=(LinearConstraint constraint);

    LinearConstraintConjunction&
    operator&=(const LinearConstraintConjunction& conj);

    [[nodiscard]]
    expressions::Expression as_expression() const;

    [[nodiscard]]
    std::string to_string(size_t ws = 0) const;

    template <typename ValueGetter>
    [[nodiscard]]
    bool evaluate(ValueGetter get_value) const
    {
        return std::all_of(begin(), end(), [&get_value](auto&& constraint) {
            return constraint.evaluate(get_value);
        });
    }
};

struct LinearConstraintDisjunction : public vector<LinearConstraint> {
    LinearConstraintDisjunction operator|=(LinearConstraint constraint);

    LinearConstraintDisjunction
    operator|=(const LinearConstraintDisjunction& disj);

    [[nodiscard]]
    expressions::Expression as_expression() const;

    [[nodiscard]]
    std::string to_string(size_t ws = 0) const;

    template <typename ValueGetter>
    [[nodiscard]]
    bool evaluate(ValueGetter get_value) const
    {
        return std::any_of(begin(), end(), [&get_value](auto&& constraint) {
            return constraint.evaluate(get_value);
        });
    }
};

struct LinearCondition : public vector<LinearConstraintConjunction> {
    LinearCondition& operator|=(const LinearCondition& other);

    LinearCondition& operator|=(const LinearConstraintDisjunction& other);

    LinearCondition& operator|=(LinearConstraint other);

    LinearCondition& operator|=(LinearConstraintConjunction other);
    [[nodiscard]]

    [[nodiscard]]
    std::string to_string(size_t ws = 0) const;

    [[nodiscard]]
    static LinearCondition from_expression(const expressions::Expression& expr);

    expressions::Expression as_expression() const;

    template <typename ValueGetter>
    [[nodiscard]]
    bool evaluate(ValueGetter get_value) const
    {
        return std::any_of(begin(), end(), [&get_value](auto&& constraint) {
            return constraint.evaluate(get_value);
        });
    }
};

[[nodiscard]]
LinearConstraintDisjunction operator!(const LinearConstraintConjunction& conj);

[[nodiscard]]
vector<LinearConstraintDisjunction> operator!(const LinearCondition& cond);

[[nodiscard]]
LinearConstraintConjunction operator&&(
    const LinearConstraintConjunction& a,
    const LinearConstraintConjunction& b);

[[nodiscard]]
LinearConstraintConjunction
operator&&(const LinearConstraint& a, const LinearConstraintConjunction& b);

[[nodiscard]]
LinearConstraintConjunction
operator&&(const LinearConstraintConjunction& a, const LinearConstraint& b);

[[nodiscard]]
LinearConstraintDisjunction operator||(
    const LinearConstraintDisjunction& a,
    const LinearConstraintDisjunction& b);

[[nodiscard]]
LinearConstraintDisjunction
operator||(const LinearConstraint& a, const LinearConstraintDisjunction& b);

[[nodiscard]]
LinearConstraintDisjunction
operator||(const LinearConstraintDisjunction& a, const LinearConstraint& b);

[[nodiscard]]
LinearCondition operator||(const LinearCondition& a, const LinearCondition& b);

[[nodiscard]]
LinearCondition operator||(const LinearConstraint& a, const LinearCondition& b);

[[nodiscard]]
LinearCondition operator||(const LinearCondition& a, const LinearConstraint& b);

[[nodiscard]]
LinearCondition
operator||(const LinearConstraintConjunction& a, const LinearCondition& b);

[[nodiscard]]
LinearCondition
operator||(const LinearCondition& a, const LinearConstraintConjunction& b);

[[nodiscard]]
LinearCondition
operator||(const LinearConstraintDisjunction& a, const LinearCondition& b);

[[nodiscard]]
LinearCondition
operator||(const LinearCondition& a, const LinearConstraintDisjunction& b);

std::ostream&
operator<<(std::ostream& out, const police::LinearConstraintDisjunction& disj);

std::ostream&
operator<<(std::ostream& out, const police::LinearConstraintConjunction& conj);

std::ostream& operator<<(std::ostream& out, const police::LinearCondition& cond);

} // namespace police
