#pragma once

#include "police/base_types.hpp"
#include "police/expressions/expression.hpp"
#include "police/linear_expression.hpp"
#include "police/macros.hpp"
#include "police/utils/hash.hpp"

namespace police {

struct LinearConstraint : public LinearCombination<size_t, real_t> {
    template <typename>
    friend struct police::hash;

    enum Type {
        LESS_EQUAL,
        GREATER_EQUAL,
        EQUAL,
    };

    explicit LinearConstraint(Type type);

    LinearConstraint(LinearCombination<size_t, real_t> lhs, real_t rhs, Type t);

    [[nodiscard]]
    static LinearConstraint
    from_expression(const expressions::Expression& expr);

    [[nodiscard]]
    static LinearConstraint
    unit_constraint(size_t var_id, Type type, real_t value);

    template <typename ValueGetter>
    [[nodiscard]]
    bool evaluate(ValueGetter get_value) const
    {
        const auto lhs =
            LinearCombination<size_t, real_t>::evaluate(std::move(get_value));
        switch (type) {
        case Type::LESS_EQUAL: return lhs <= rhs;
        case Type::EQUAL: return lhs == rhs;
        case Type::GREATER_EQUAL: return lhs >= rhs;
        }
        POLICE_UNREACHABLE();
    }

    [[nodiscard]]
    LinearConstraint operator-() const;

    [[nodiscard]]
    expressions::Expression as_expression() const;

    [[nodiscard]]
    std::string to_string() const;

    [[nodiscard]]
    bool operator==(const LinearConstraint& c) const
    {
        return type == c.type && c.rhs == rhs &&
               LinearCombination::operator==(c);
    }

    real_t rhs{};
    Type type;
};

LinearConstraint
less_equal(const LinearExpression& lhs, const LinearExpression& rhs);
LinearConstraint less_equal(const LinearExpression& lhs, real_t rhs);

LinearConstraint
greater_equal(const LinearExpression& lhs, const LinearExpression& rhs);
LinearConstraint greater_equal(const LinearExpression& lhs, real_t rhs);

LinearConstraint
equal(const LinearExpression& lhs, const LinearExpression& rhs);
LinearConstraint equal(const LinearExpression& lhs, real_t rhs);

template <>
struct hash<LinearConstraint> {
    [[nodiscard]]
    std::size_t operator()(const LinearConstraint& c) const
    {
        return police::hash_combine(
            c.elements_hash(),
            get_hash(c.rhs),
            get_hash(static_cast<int_t>(c.type)));
    }
};

std::ostream&
operator<<(std::ostream& out, const police::LinearConstraint& constraint);

} // namespace police
