#include "police/lp_gurobi.hpp"

#ifdef POLICE_GUROBI

#include "police/arguments.hpp"
#include "police/constants.hpp"
#include "police/jani/parser/types.hpp"
#include "police/lp.hpp"
#include "police/macros.hpp"
#include "police/option.hpp"
#include "police/storage/variable_space.hpp"

#include <algorithm>
#include <cassert>
#include <gurobi_c++.h>
#include <gurobi_c.h>
#include <iterator>
#include <limits>
#include <type_traits>

#define GUROBI_EXCEPTION(e)                                                    \
    POLICE_RUNTIME_ERROR(                                                       \
        "Gurobi has thrown exception \"" << e.getMessage() << "\" with code "  \
                                         << e.getErrorCode() << " at line "    \
                                         << __LINE__ << " of " << __FILE__)

#define GUROBI_IIS_CORE 0

namespace police {

namespace {
GRBEnv& get_gurobi_environment()
{
    thread_local std::unique_ptr<GRBEnv> env_ = nullptr;
    if (env_ == nullptr) {
        try {
            env_.reset(new GRBEnv());
            env_->set(GRB_IntParam_OutputFlag, 0);
            env_->set(GRB_IntParam_Threads, 1);
            // env_->set(GRB_DoubleParam_FeasibilityTol, 1e-9);
            // env_->set(GRB_DoubleParam_IntFeasTol, 1e-8);
            env_->start();
            env_->message("Gurobi environment setup.");
        } catch (const GRBException& err) {
            GUROBI_EXCEPTION(err);
        }
    }
    return *env_;
}
} // namespace

struct GurobiLP::GurobiInternals {
    GurobiInternals()
        : model(get_gurobi_environment())
    {
    }

    GRBModel model;
};

void GurobiLP::GRBDeleter::operator()(GurobiLP::GurobiInternals* model) const
{
    delete (model);
}

namespace {
struct GurobiModel {
    GurobiModel(
        GRBModel* model,
        const vector<size_t>* var_remap,
        const VariableSpace* vspace)
        : model_(model)
        , var_remap_(var_remap)
        , vspace_(vspace)
    {
    }

    [[nodiscard]]
    Value get_value(size_t var_idx) const
    {
        assert(var_idx < var_remap_->size());
        const size_t actual_idx = var_remap_->at(var_idx);
        const auto& var_type = vspace_->at(actual_idx).type;
        try {
            const auto val = model_->getVar(actual_idx).get(GRB_DoubleAttr_X);
            return std::visit(
                [&](auto&& t) {
                    using T = std::decay_t<decltype(t)>;
                    if (std::is_same_v<T, BoolType>) {
                        return Value(
                            static_cast<int_t>(val + LP_PRECISION) >= 1);
                    } else if (
                        std::is_same_v<T, IntegerType> ||
                        std::is_same_v<T, jani::parser::BoundedIntType>) {
                        return Value(static_cast<int_t>(val + LP_PRECISION));
                    } else {
                        return Value(val);
                    }
                },
                var_type);
        } catch (GRBException& e) {
            GUROBI_EXCEPTION(e);
        }
    }

    [[nodiscard]]
    size_t size() const
    {
        return var_remap_->size();
    }

private:
    GRBModel* model_;
    const vector<size_t>* var_remap_;
    const VariableSpace* vspace_;
};

} // namespace

GurobiLP::GurobiLP(LPOptimizationKind sense)
    : LP(sense)
    , gurobi_(nullptr)
{
    try {
        gurobi_.reset(new GurobiInternals());
    } catch (const GRBException& err) {
        GUROBI_EXCEPTION(err);
    }
    set_sense(sense);
}

void GurobiLP::set_sense(LPOptimizationKind sense)
{
    try {
        gurobi_->model.set(
            GRB_IntAttr_ModelSense,
            sense == LPOptimizationKind::MAXIMIZE ? -1 : 1);
    } catch (const GRBException& err) {
        GUROBI_EXCEPTION(err);
    }
}

LPOptimizationKind GurobiLP::get_sense() const
{
    try {
        return gurobi_->model.get(GRB_IntAttr_ModelSense) < 0
                   ? LPOptimizationKind::MINIMIZE
                   : LPOptimizationKind::MAXIMIZE;
    } catch (const GRBException& err) {
        GUROBI_EXCEPTION(err);
    }
}

real_t GurobiLP::get_infinity() const
{
    return GRB_INFINITY;
}

void GurobiLP::push_snapshot()
{
    gurobi_->model.update();
    snapshots_.emplace_back(
        num_vars_,
        get_num_variables(),
        get_num_lin_constraints(),
        get_num_gen_constraints(),
        indicator_vars_.size());
}

void GurobiLP::pop_snapshot()
{
    assert(!snapshots_.empty());
    auto snapshot = snapshots_.back();
    snapshots_.pop_back();
    try {
        gurobi_->model.update();

        assert(get_num_variables() >= snapshot.total_vars);
        for (auto i = get_num_variables(); i > snapshot.total_vars; --i) {
            gurobi_->model.remove(gurobi_->model.getVar(i - 1));
        }
        vspace_.erase(vspace_.begin() + snapshot.total_vars, vspace_.end());
        objective_coefs_.erase(
            objective_coefs_.begin() + snapshot.total_vars,
            objective_coefs_.end());

        var_remap_.erase(var_remap_.begin() + snapshot.vars, var_remap_.end());
        num_vars_ = snapshot.vars;

        assert(get_num_lin_constraints() >= snapshot.lin_constraints);
        assert(linear_constraints_indices_.size() == get_num_lin_constraints());
        for (auto i = get_num_lin_constraints(); i > snapshot.lin_constraints;
             --i) {
            gurobi_->model.remove(
                gurobi_->model.getConstr(linear_constraints_indices_[i - 1]));
        }
        linear_constraints_indices_.resize(snapshot.lin_constraints);

        assert(get_num_gen_constraints() >= snapshot.gen_constraints);
        if (get_num_gen_constraints() - snapshot.gen_constraints > 0) {
            auto* cc = gurobi_->model.getGenConstrs();
            for (auto i = get_num_gen_constraints();
                 i > snapshot.gen_constraints;
                 --i) {
                gurobi_->model.remove(cc[i - 1]);
            }
            delete[] (cc);
        }

        assert(indicator_constraints_.size() == indicator_vars_.size());
        for (size_t i = indicator_vars_.size();
             i > snapshot.indicator_constraints;
             --i) {
            indicator_constraints_.erase(indicator_constraints_.find(
                indicator_constraints_.data()->at(i - 1)));
        }
        indicator_constraints_.defragment(true);
        indicator_vars_.erase(
            indicator_vars_.begin() + snapshot.indicator_constraints,
            indicator_vars_.end());

        gurobi_->model.update();
    } catch (GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

LP::model_type GurobiLP::get_model() const
{
    return GurobiModel(&gurobi_->model, &var_remap_, &vspace_);
}

real_t GurobiLP::get_objective_value() const
{
    try {
        return gurobi_->model.get(GRB_DoubleAttr_ObjVal);
    } catch (GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

std::string GurobiLP::get_name() const
{
    return "Gurobi";
}

size_t GurobiLP::num_constraints() const
{
    return get_num_gen_constraints() + get_num_lin_constraints();
}

size_t GurobiLP::get_num_lin_constraints() const
{
    try {
        return gurobi_->model.get(GRB_IntAttr_NumConstrs);
    } catch (GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}
size_t GurobiLP::get_num_gen_constraints() const
{
    try {
        return gurobi_->model.get(GRB_IntAttr_NumGenConstrs);
    } catch (GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

size_t GurobiLP::get_num_variables() const
{
    try {
        return gurobi_->model.get(GRB_IntAttr_NumVars);
    } catch (GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

size_t GurobiLP::num_variables() const
{
    return num_vars_;
}

LPStatus GurobiLP::solve()
{
    try {
        gurobi_->model.set(GRB_IntParam_Presolve, 1);
        gurobi_->model.optimize();
        int optimstatus = gurobi_->model.get(GRB_IntAttr_Status);
        if (optimstatus == GRB_INF_OR_UNBD) {
            gurobi_->model.set(GRB_IntParam_Presolve, 0);
            gurobi_->model.optimize();
            optimstatus = gurobi_->model.get(GRB_IntAttr_Status);
        }
        switch (optimstatus) {
        case GRB_OPTIMAL: return LPStatus::OPTIMAL;
        case GRB_INFEASIBLE: return LPStatus::INFEASIBLE;
        case GRB_UNBOUNDED: return LPStatus::UNBOUNDED;
        default:
            POLICE_RUNTIME_ERROR(
                "unknown gurobi solution status " << optimstatus);
        }
    } catch (GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

namespace {
char get_gurobi_var_type(LPVariable::Type t)
{
    switch (t) {
    case LPVariable::Type::REAL: return GRB_CONTINUOUS;
    case LPVariable::Type::INT: return GRB_INTEGER;
    case LPVariable::Type::BOOL: return GRB_BINARY;
    }
    POLICE_UNREACHABLE();
}
} // namespace

LP::variable_ref GurobiLP::add_variable(const variable_type& var)
{
    try {
        auto gvar = gurobi_->model.addVar(
            var.lower_bound,
            var.upper_bound,
            var.obj_coef,
            get_gurobi_var_type(var.type));
        notify_new_variable(gvar);
        var_remap_.push_back(gvar.index());
        objective_coefs_.push_back(var.obj_coef);
        switch (var.type) {
        case variable_type::Type::BOOL:
            vspace_.add_variable("", BoolType());
            break;
        case variable_type::Type::INT:
            vspace_.add_variable("", IntegerType());
            break;
        case variable_type::Type::REAL:
            vspace_.add_variable("", RealType());
            break;
        default: POLICE_UNREACHABLE();
        }
        assert(vspace_.size() == get_num_variables());
        ++num_vars_;
        return num_vars_ - 1;
    } catch (GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

void GurobiLP::set_variable_lower_bound(variable_ref var_id, real_t lb)
{
    try {
        assert(var_id < var_remap_.size());
        auto var = gurobi_->model.getVar(var_remap_[var_id]);
        if (lb == -std::numeric_limits<real_t>::infinity()) {
            var.set(GRB_DoubleAttr_LB, -get_infinity());
        } else {
            var.set(GRB_DoubleAttr_LB, lb);
        }
    } catch (const GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

void GurobiLP::set_variable_upper_bound(variable_ref var_id, real_t ub)
{
    try {
        assert(var_id < var_remap_.size());
        auto var = gurobi_->model.getVar(var_remap_[var_id]);
        if (ub == std::numeric_limits<real_t>::infinity()) {
            var.set(GRB_DoubleAttr_UB, get_infinity());
        } else {
            var.set(GRB_DoubleAttr_UB, ub);
        }
    } catch (const GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

namespace {
GRBLinExpr get_gurobi_lin_expr(
    GRBModel& model,
    const size_t* vars,
    const real_t* coefs,
    size_t size,
    const vector<size_t>& var_remap)
{
    GRBLinExpr expr(0.0);
    for (auto j = 0u; j < size; ++j) {
        assert(vars[j] < var_remap.size());
        expr += GRBLinExpr(model.getVar(var_remap[vars[j]]), coefs[j]);
    }
    return expr;
}

char get_lin_constraint_sense(LinearConstraint::Type type)
{
    switch (type) {
    case LinearConstraint::Type::EQUAL: return GRB_EQUAL;
    case LinearConstraint::Type::LESS_EQUAL: return GRB_LESS_EQUAL;
    case LinearConstraint::Type::GREATER_EQUAL: return GRB_GREATER_EQUAL;
    }
    POLICE_UNREACHABLE();
}
} // namespace

LP::constraint_ref GurobiLP::add_constraint(const LinearConstraint& constraint)
{
    try {
        auto gconstr = gurobi_->model.addConstr(
            get_gurobi_lin_expr(
                gurobi_->model,
                constraint.refs(),
                constraint.coefs(),
                constraint.size(),
                var_remap_),
            get_lin_constraint_sense(constraint.type),
            constraint.rhs);
        notify_new_constraint(gconstr);
        linear_constraints_indices_.push_back(gconstr.index());
        return gconstr.index();
    } catch (GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

LP::constraint_ref
GurobiLP::add_constraint(const disjunctive_constraint_type& disj)
{
    try {
        GRBLinExpr expr(0.);
        for (auto i = 0u; i < disj.size(); ++i) {
            auto var_id = create_indicator_constraint(disj[i]);
            auto var = gurobi_->model.getVar(var_id);
            expr += var;
        }
        auto enforce_constraint =
            gurobi_->model.addConstr(expr, GRB_GREATER_EQUAL, 1);
        notify_new_constraint(enforce_constraint);
        linear_constraints_indices_.push_back(enforce_constraint.index());
        return -1;
    } catch (GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

LP::constraint_ref
GurobiLP::add_constraint(const indicator_constraint_type& constraint)
{
    assert(constraint.indicator_var < var_remap_.size());
    auto ivar = var_remap_[constraint.indicator_var];
    add_indicator_constraint(
        ivar,
        constraint.indicator_value,
        constraint.constraint);
    return -1;
}

void GurobiLP::bind_indicator_value(size_t var, bool val, size_t indicator)
{
    const auto& [ivar, ival] = indicator_vars_[indicator];
    if (var != ivar) {
        try {
            GRBLinExpr expr(0.);
            expr += GRBLinExpr(gurobi_->model.getVar(ivar), 1.);
            char sense = GRB_EQUAL;
            double rhs = 0.;
            if (val == ival) {
                expr -= GRBLinExpr(gurobi_->model.getVar(var), 1.);
            } else {
                expr += GRBLinExpr(gurobi_->model.getVar(var), 1.);
                rhs = 1.;
            }
            auto grbconstr =
                gurobi_->model.addConstr(std::move(expr), sense, rhs);
            notify_new_constraint(grbconstr);
            linear_constraints_indices_.push_back(grbconstr.index());
        } catch (const GRBException& err) {
            GUROBI_EXCEPTION(err);
        }
    } else {
        assert(val == ival);
    }
}

LP::variable_ref
GurobiLP::create_indicator_constraint(const linear_constraint_type& constraint)
{
    auto pos = get_indicator_constraint_index(constraint);
    assert(pos <= indicator_vars_.size());
    if (pos == indicator_vars_.size()) {
        auto var = gurobi_->model.addVar(0, 1, 0, GRB_CONTINUOUS);
        notify_new_variable(var);
        vspace_.add_variable("", RealType());
        objective_coefs_.push_back(0);
        add_indicator_constraint(var.index(), true, constraint);
        indicator_vars_.emplace_back(var.index(), true);
        return var.index();
    } else {
        const auto& [var, val] = indicator_vars_[pos];
        if (val) {
            return var;
        }
        auto new_var = gurobi_->model.addVar(0, 1, 0, GRB_CONTINUOUS);
        notify_new_variable(new_var);
        vspace_.add_variable("", RealType());
        objective_coefs_.push_back(0);
        bind_indicator_value(new_var.index(), true, pos);
        return new_var.index();
    }
}

size_t GurobiLP::get_indicator_constraint_index(
    [[maybe_unused]] const linear_constraint_type& constraint)
{
    if (constraint.type == linear_constraint_type::GREATER_EQUAL) {
        return indicator_constraints_.insert(-constraint).first->second;
    } else {
        return indicator_constraints_.insert(constraint).first->second;
    }
}

void GurobiLP::add_indicator_constraint(
    size_t var,
    bool val,
    const linear_constraint_type& constraint)
{
    try {
        GRBLinExpr expr = get_gurobi_lin_expr(
            gurobi_->model,
            constraint.refs(),
            constraint.coefs(),
            constraint.size(),
            var_remap_);
        auto gc = gurobi_->model.addGenConstrIndicator(
            gurobi_->model.getVar(var),
            val,
            std::move(expr),
            get_lin_constraint_sense(constraint.type),
            constraint.rhs);
        notify_new_constraint(gc);
    } catch (GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

LP::constraint_ref
GurobiLP::add_constraint(const max_constraint_type& constraint)
{
    try {
        assert(constraint.y < var_remap_.size());
        vector<GRBVar> vars;
        vars.reserve(constraint.elements.size());
        for (const auto& var_id : constraint.elements) {
            assert(var_id < var_remap_.size());
            vars.push_back(gurobi_->model.getVar(var_remap_[var_id]));
        }
        auto res_var = gurobi_->model.getVar(var_remap_[constraint.y]);
        double constant =
            constraint.c == -std::numeric_limits<real_t>::infinity()
                ? -get_infinity()
                : constraint.c;
        auto c = gurobi_->model.addGenConstrMax(
            std::move(res_var),
            vars.data(),
            vars.size(),
            constant);
        notify_new_constraint(c);
        return -1;
    } catch (GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

#if GUROBI_IIS_CORE
LPStatus GurobiLP::solve(const vector<linear_constraint_type>& assumptions)
{
    try {
        gurobi_->model.update();
        unsat_core_.clear();
        vector<GRBConstr> constraints;
        constraints.reserve(assumptions.size());
        std::transform(
            assumptions.begin(),
            assumptions.end(),
            std::back_inserter(constraints),
            [&](const linear_constraint_type& constraint) {
                auto expr = get_gurobi_lin_expr(
                    gurobi_->model,
                    constraint.refs(),
                    constraint.coefs(),
                    constraint.size(),
                    var_remap_);
                return gurobi_->model.addConstr(
                    std::move(expr),
                    get_lin_constraint_sense(constraint.type),
                    constraint.rhs);
            });
        LPStatus result = SOLVABLE;
        try {
            gurobi_->model.computeIIS();
            for (size_t i = 0; i < constraints.size(); ++i) {
                if (constraints[i].get(GRB_IntAttr_IISConstr)) {
                    unsat_core_.push_back(assumptions[i]);
                }
            }
            result = INFEASIBLE;
        } catch (const GRBException& err) {
            if (err.getErrorCode() != GRB_ERROR_IIS_NOT_INFEASIBLE) {
                throw err;
            }
        }
        for (int i = constraints.size() - 1; i >= 0; --i) {
            gurobi_->model.remove(constraints[i]);
        }
        return result;
    } catch (const GRBException& err) {
        GUROBI_EXCEPTION(err);
    }
}
#else
LPStatus GurobiLP::solve(const vector<linear_constraint_type>& assumptions)
{
    unsat_core_.clear();
    if (assumptions.empty()) return solve();
    try {
#ifndef NDEBUG
        const size_t old_num_vars = get_num_variables();
        const size_t old_num_constraints = get_num_lin_constraints();
        const size_t old_num_gen_constraints = get_num_gen_constraints();
#endif

        const int sense = clear_objective();

        vector<GRBVar> indicators;
        vector<GRBGenConstr> constraints;
        auto is_satisfied = [](GRBVar& var) {
            return static_cast<int>(var.get(GRB_DoubleAttr_X) + LP_PRECISION) ==
                   1;
        };

        indicators.reserve(assumptions.size());
        std::transform(
            assumptions.begin(),
            assumptions.end(),
            std::back_inserter(indicators),
            [&](const auto&) {
                return gurobi_->model.addVar(0, 1, 1, GRB_BINARY);
            });
        gurobi_->model.set(GRB_IntAttr_ModelSense, GRB_MAXIMIZE);
        gurobi_->model.update();

        constraints.reserve(assumptions.size());
        for (auto i = 0u; i < assumptions.size(); ++i) {
            const auto& constraint = assumptions[i];
            auto& var = indicators[i];
            auto expr = get_gurobi_lin_expr(
                gurobi_->model,
                constraint.refs(),
                constraint.coefs(),
                constraint.size(),
                var_remap_);
            constraints.push_back(gurobi_->model.addGenConstrIndicator(
                var,
                1,
                std::move(expr),
                get_lin_constraint_sense(constraint.type),
                constraint.rhs));
        }

        gurobi_->model.optimize();
        auto status = gurobi_->model.get(GRB_IntAttr_Status);
        LPStatus res = SOLVABLE;

        switch (status) {
        case GRB_OPTIMAL: {
            const size_t num_sat =
                static_cast<size_t>(get_objective_value() + LP_PRECISION);
            assert(num_sat <= assumptions.size());
            assert(
                std::count_if(
                    indicators.begin(),
                    indicators.end(),
                    is_satisfied) == num_sat);
            if (num_sat == assumptions.size()) {
                res = LPStatus::SOLVABLE;
            } else {
                vector<GRBConstr> constraints;
                res = LPStatus::INFEASIBLE;
                while (status == GRB_OPTIMAL) {
                    auto var = std::find_if_not(
                        indicators.begin(),
                        indicators.end(),
                        is_satisfied);
                    assert(var != indicators.end());
                    GRBLinExpr expr(0.0);
                    expr += *var;
                    constraints.push_back(gurobi_->model.addConstr(
                        std::move(expr),
                        GRB_GREATER_EQUAL,
                        1.));
                    unsat_core_.push_back(
                        assumptions[std::distance(indicators.begin(), var)]);
                    gurobi_->model.optimize();
                    status = gurobi_->model.get(GRB_IntAttr_Status);
                }
                assert(status == GRB_INFEASIBLE);
                for (int i = constraints.size() - 1; i >= 0; --i) {
                    gurobi_->model.remove(constraints[i]);
                }
            }
            break;
        }
        case GRB_INFEASIBLE: res = LPStatus::INFEASIBLE; break;
        case GRB_UNBOUNDED: res = LPStatus::UNBOUNDED; break;
        default:
            POLICE_RUNTIME_ERROR(
                "gurobi exited with unknown optimization status " << status);
        }

        for (int i = constraints.size() - 1; i >= 0; --i) {
            gurobi_->model.remove(constraints[i]);
        }
        for (int i = indicators.size() - 1; i >= 0; --i) {
            gurobi_->model.remove(indicators[i]);
        }
        restore_objective(sense);

#ifndef NDEBUG
        gurobi_->model.update();
        assert(old_num_vars == get_num_variables());
        assert(old_num_constraints == get_num_lin_constraints());
        assert(old_num_gen_constraints == get_num_gen_constraints());
#endif

        return res;
    } catch (const GRBException& err) {
        GUROBI_EXCEPTION(err);
    }
}
#endif

void GurobiLP::restore_objective(int sense)
{
    try {
        for (int i = objective_coefs_.size() - 1; i >= 0; --i) {
            gurobi_->model.getVar(i).set(
                GRB_DoubleAttr_Obj,
                objective_coefs_[i]);
        }
        gurobi_->model.set(GRB_IntAttr_ModelSense, sense);
    } catch (const GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

int GurobiLP::clear_objective()
{
    try {
        for (int i = objective_coefs_.size() - 1; i >= 0; --i) {
            gurobi_->model.getVar(i).set(GRB_DoubleAttr_Obj, 0);
        }
        return gurobi_->model.get(GRB_IntAttr_ModelSense);
    } catch (const GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

vector<LP::linear_constraint_type> GurobiLP::get_unsat_core() const
{
    return unsat_core_;
}

#if 0
void GurobiLP::change_obj_coefs(
    std::vector<variable_ref>&& variables,
    std::vector<real_t>&& coefs)
{
    try {
        for (auto i = 0u; i < variables.size(); ++i) {
            gurobi_->model.getVar(variables[i]).set(GRB_DoubleAttr_Obj, coefs[i]);
        }
    } catch (GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

void GurobiLP::change_coefs(
    std::vector<constraint_ref>&& constraints,
    std::vector<variable_ref>&& variables,
    std::vector<real_t>&& coefs)
{
    try {
        for (auto i = 0u; i < variables.size(); ++i) {
            gurobi_->model.chgCoeff(
                gurobi_->model.getConstr(linear_constraints_indices_[constraints[i]]),
                gurobi_->model.getVar(variables[i]),
                coefs[i]);
        }
    } catch (GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

void GurobiLP::change_bounds(
    std::vector<constraint_ref>&& constraints,
    std::vector<real_t>&& bounds)
{
    try {
        for (auto i = 0u; i < constraints.size(); ++i) {
            auto constraint = gurobi_->model.getConstr(linear_constraints_indices_[constraints[i]]);
            constraint.set(GRB_DoubleAttr_RHS, bounds[i]);
        }
    } catch (GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}
#endif

namespace {
void dump_linear_expression(std::ostream& out, const GRBLinExpr& expr)
{
    if (expr.size() == 0) {
        out << expr.getConstant();
    } else {
        out << expr.getCoeff(0) << "x" << expr.getVar(0).index();
        for (auto j = 1u; j < expr.size(); ++j) {
            const auto coef = expr.getCoeff(j);
            const auto var_id = expr.getVar(j).index();
            if (coef < 0.) {
                out << " - " << (-coef) << "x" << var_id;
            } else {
                out << " + " << coef << "x" << var_id;
            }
        }
        if (expr.getConstant() < 0.) {
            out << " - " << (-expr.getConstant());
        } else if (expr.getConstant() > 0.) {
            out << " + " << expr.getConstant();
        }
    }
}

void dump_objective(std::ostream& out, const GRBModel& model)
{
    const auto num_vars = model.get(GRB_IntAttr_NumVars);
    const auto sense = model.get(GRB_IntAttr_ModelSense);
    out << (sense > 0 ? "minimize" : "maximize") << " ";
    bool first = true;
    for (auto var_id = 0; var_id < num_vars; ++var_id) {
        const auto var = model.getVar(var_id);
        const auto coef = var.get(GRB_DoubleAttr_Obj);
        if (coef < 0.) {
            if (first) {
                out << coef << "x" << var_id;
            } else {
                out << " - " << (-coef) << "x" << var_id;
            }
            first = false;
        } else if (coef > 0.) {
            out << (first ? "" : " + ") << coef << "x" << var_id;
            first = false;
        }
    }
    if (first) {
        out << "0";
    }
    out << "\n";
}

void dump_variable_bounds(std::ostream& out, const GRBModel& model)
{
    const auto infinity = GRB_INFINITY;
    const auto num_vars = model.get(GRB_IntAttr_NumVars);
    for (auto var = 0; var < num_vars; ++var) {
        const auto v = model.getVar(var);
        const auto lb = v.get(GRB_DoubleAttr_LB);
        const auto ub = v.get(GRB_DoubleAttr_UB);
        const auto type = v.get(GRB_CharAttr_VType);
        out << "x" << var << ": ";
        if (type == GRB_INTEGER) {
            out << "integer";
        } else if (type == GRB_CONTINUOUS) {
            out << "real";
        } else if (type == GRB_BINARY) {
            out << "binary";
        } else {
            out << "unknown(" << type << ")";
        }
        if (lb != -infinity) {
            if (ub != infinity) {
                out << " where " << lb << " <= x" << var << " <= " << ub
                    << "\n";
            } else {
                out << " where " << "x" << var << " >= " << lb << "\n";
            }
        } else if (ub != infinity) {
            out << " where " << "x" << var << " <= " << ub << "\n";
        } else {
            out << "\n";
        }
    }
}

void dump_linear_constraints(std::ostream& out, GRBModel& model)
{
    const auto num_constraints = model.get(GRB_IntAttr_NumConstrs);
    for (int i = 0; i < num_constraints; ++i) {
        const auto& c = model.getConstr(i);
        const auto rhs = c.get(GRB_DoubleAttr_RHS);
        const auto expr = model.getRow(c);
        dump_linear_expression(out, expr);
        const auto sense = c.get(GRB_CharAttr_Sense);
        if (sense == GRB_LESS_EQUAL) {
            out << " <= ";
        } else if (sense == GRB_GREATER_EQUAL) {
            out << " >= ";
        } else {
            out << " == ";
        }
        out << rhs << "\n";
    }
}

void dump_disjunctive_constraints(std::ostream& out, GRBModel& model)
{
    const auto num_gen_constraints = model.get(GRB_IntAttr_NumGenConstrs);
    const auto* constraints = model.getGenConstrs();
    for (int i = 0; i < num_gen_constraints; ++i) {
        const auto& c = constraints[i];
        const auto type = c.get(GRB_IntAttr_GenConstrType);
        if (type == GRB_GENCONSTR_OR) {
            int num_vars = 0;
            model.getGenConstrOr(c, nullptr, nullptr, &num_vars);
            GRBVar res;
            vector<GRBVar> vars(num_vars);
            model.getGenConstrOr(c, &res, vars.data(), nullptr);
            out << "x" << res.index() << " = ";
            for (int j = 0; j < num_vars; ++j) {
                out << (j > 0 ? " OR " : "") << "x" << vars[j].index();
            }
            if (num_vars == 0) {
                out << "0";
            }
            out << "\n";
        }
    }
    delete[] (constraints);
}

void dump_max_constraints(std::ostream& out, GRBModel& model)
{
    const auto num_gen_constraints = model.get(GRB_IntAttr_NumGenConstrs);
    const auto* constraints = model.getGenConstrs();
    for (int i = 0; i < num_gen_constraints; ++i) {
        const auto& c = constraints[i];
        const auto type = c.get(GRB_IntAttr_GenConstrType);
        if (type == GRB_GENCONSTR_MAX) {
            int num_vars = 0;
            model.getGenConstrMax(c, nullptr, nullptr, &num_vars, nullptr);
            GRBVar res;
            vector<GRBVar> vars(num_vars);
            double constant = 0.;
            model.getGenConstrMax(c, &res, vars.data(), nullptr, &constant);
            out << "x" << res.index() << " = max(";
            for (int j = 0; j < num_vars; ++j) {
                out << (j > 0 ? ", " : "") << "x" << vars[j].index();
            }
            if (num_vars == 0) {
                out << "0";
            }
            if (constant != -GRB_INFINITY) {
                out << ", " << constant;
            }
            out << ")\n";
        }
    }
    delete[] (constraints);
}

void dump_indicator_constraints(std::ostream& out, GRBModel& model)
{
    const auto num_gen_constraints = model.get(GRB_IntAttr_NumGenConstrs);
    const auto* constraints = model.getGenConstrs();
    for (int i = 0; i < num_gen_constraints; ++i) {
        const auto& c = constraints[i];
        const auto type = c.get(GRB_IntAttr_GenConstrType);
        if (type == GRB_GENCONSTR_INDICATOR) {
            GRBVar res;
            int val;
            GRBLinExpr expr;
            char sense;
            double rhs;
            model.getGenConstrIndicator(c, &res, &val, &expr, &sense, &rhs);
            out << "x" << res.index() << " = " << val << " => ";
            dump_linear_expression(out, expr);
            switch (sense) {
            case GRB_LESS_EQUAL: out << " <= "; break;
            case GRB_GREATER_EQUAL: out << " >= "; break;
            case GRB_EQUAL: out << " == "; break;
            }
            out << rhs << "\n";
        }
    }
    delete[] (constraints);
}

} // namespace

void GurobiLP::dump(std::ostream& out) const
{
    try {
        gurobi_->model.update();
        dump_objective(out, gurobi_->model);
        dump_variable_bounds(out, gurobi_->model);
        dump_linear_constraints(out, gurobi_->model);
        dump_max_constraints(out, gurobi_->model);
        dump_disjunctive_constraints(out, gurobi_->model);
        dump_indicator_constraints(out, gurobi_->model);
        gurobi_->model.write("gurobi.lp");
    } catch (const GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

void GurobiLP::dump_model(std::ostream& out) const
{
    try {
        for (size_t var = 0; var < get_num_variables(); ++var) {
            const auto val = gurobi_->model.getVar(var).get(GRB_DoubleAttr_X);
            out << "x" << var << " = " << val << "\n";
        }
        out << std::flush;
    } catch (const GRBException& e) {
        GUROBI_EXCEPTION(e);
    }
}

void GurobiLP::notify_new_variable(GRBVar& var)
{
    try {
        gurobi_->model.update();
        var.set(GRB_IntAttr_IISLBForce, 1);
        var.set(GRB_IntAttr_IISUBForce, 1);
    } catch (const GRBException& err) {
        GUROBI_EXCEPTION(err);
    }
}

void GurobiLP::notify_new_constraint(GRBConstr& constraint)
{
    try {
        gurobi_->model.update();
        constraint.set(GRB_IntAttr_IISConstrForce, 1);
    } catch (const GRBException& err) {
        GUROBI_EXCEPTION(err);
    }
}

void GurobiLP::notify_new_constraint([[maybe_unused]] GRBGenConstr& constraint)
{
    try {
        // Gurobi bug: IISGenConstrForce not set correctly
        // gurobi_->model.update();
        // constraint.set(GRB_IntAttr_IISGenConstrForce, 1);
    } catch (const GRBException& err) {
        GUROBI_EXCEPTION(err);
    }
}

void GurobiLP::notify_new_constraint(GRBSOS& constraint)
{
    try {
        gurobi_->model.update();
        constraint.set(GRB_IntAttr_IISSOSForce, 1);
    } catch (const GRBException& err) {
        GUROBI_EXCEPTION(err);
    }
}

namespace {
PointerOption<LPFactory> _option("gurobi", [](const Arguments&) {
    return std::make_shared<GurobiLPFactory>();
});
} // namespace

} // namespace police

#undef GUROBI_EXCEPTION
#undef GUROBI_IIS_CORE

#endif
