#pragma once

#include "police/lp.hpp"
#include "police/lp_factory.hpp"

#if POLICE_Z3
#include "police/_bits/z3_optimizer.hpp"
#endif

namespace police {

class Z3LP final : public LP {
public:
    explicit Z3LP(LPOptimizationKind opt);

    void set_sense(LPOptimizationKind sense) override;

    variable_ref add_variable(const variable_type& var) override;

    constraint_ref
    add_constraint(const linear_constraint_type& constraint) override;

    constraint_ref
    add_constraint(const disjunctive_constraint_type& constraint) override;

    constraint_ref
    add_constraint(const indicator_constraint_type& constraint) override;

    constraint_ref
    add_constraint(const max_constraint_type& constraint) override;

    void set_variable_lower_bound(variable_ref var, real_t lb) override;

    void set_variable_upper_bound(variable_ref var, real_t lb) override;

    [[nodiscard]]
    size_t num_constraints() const override;

    [[nodiscard]]
    size_t num_variables() const override;

    [[nodiscard]]
    optimization_kind get_sense() const override;

    [[nodiscard]]
    LPStatus solve() override;

    [[nodiscard]]
    LPStatus solve(const vector<linear_constraint_type>& assumptions) override;

    [[nodiscard]]
    vector<linear_constraint_type> get_unsat_core() const override;

    void push_snapshot() override;

    void pop_snapshot() override;

    [[nodiscard]]
    model_type get_model() const override;

    [[nodiscard]]
    real_t get_objective_value() const override;

    [[nodiscard]]
    std::string get_name() const override;

    [[nodiscard]]
    real_t get_infinity() const override;

    void dump(std::ostream& out) const override;

    void dump_model(std::ostream& out) const override;

private:
#if POLICE_Z3
    void set_variable_bounds();
    void add_objective();
    void clear_optimizer_state();

    vector<real_t> objective_coefs_;
    vector<real_t> var_lbs_;
    vector<real_t> var_ubs_;
    LPOptimizationKind sense_;
    std::unique_ptr<Z3Optimizer> optimizer_;
    bool dirty_ = false;
#endif
};

class Z3LPFactory final : public LPFactory {
public:
    [[nodiscard]]
    LP* make(LPOptimizationKind opt) const override
    {
        return new Z3LP(opt);
    }
};

} // namespace police
