#pragma once

#include <cassert>  // IWYU pragma: export
#include <iostream> // IWYU pragma: export

namespace police {
class ExitCode {
public:
    enum Code {
        INPUT_ERROR = 10,
        CONFIG_ERROR = 11,
        UNSUPPORTED = 12,
        INVALID_ARGUMENT = 13,
        RUNTIME_ERROR = 20,
        INTERNAL_ERROR = 99,
    };

    ExitCode(Code code)
        : code(code)
    {
    }

    operator int() const { return static_cast<int>(code); }

    Code code;
};
} // namespace police

#define POLICE_PRINT_ERROR(x) std::cerr << "error: " << x << std::endl

#define POLICE_WARNING(x) std::cerr << "warning: " << x << std::endl

#define POLICE_INTERNAL_ERROR(x)                                                \
    {                                                                          \
        std::cerr << "internal error: " << x << std::endl;                     \
        std::exit(ExitCode::INTERNAL_ERROR);                                   \
    }

#define POLICE_EXIT_INVALID_INPUT(x)                                            \
    {                                                                          \
        std::cerr << "invalid input: " << x << std::endl;                      \
        std::exit(static_cast<int>(police::ExitCode::INPUT_ERROR));             \
    }

#define POLICE_MISSING_DEPENDENCY(x)                                            \
    {                                                                          \
        std::cerr << "error: police has been compiled without " << x            \
                  << " support." << std::endl;                                 \
        std::exit(static_cast<int>(police::ExitCode::CONFIG_ERROR));            \
    }

#define POLICE_NOT_SUPPORTED(x)                                                 \
    {                                                                          \
        std::cerr << "error: " << x << " are not supported yet." << std::endl; \
        std::exit(static_cast<int>(police::ExitCode::UNSUPPORTED));             \
    }

#define POLICE_RUNTIME_ERROR(x)                                                 \
    {                                                                          \
        std::cerr << "runtime error: " << x << std::endl;                      \
        std::exit(static_cast<int>(police::ExitCode::RUNTIME_ERROR));           \
    }

#define POLICE_UNKNOWN_ARGUMENT(arg)                                            \
    {                                                                          \
        std::cerr << "error: unknown argument " << arg << "." << std::endl;    \
        std::exit(ExitCode::INVALID_ARGUMENT);                                 \
    }

#define POLICE_INVALID_ARGUMENT(arg, reason)                                    \
    {                                                                          \
        std::cerr << "error: invalid arguments! Failed parsing " << arg        \
                  << ": " << reason << std::endl;                              \
        std::exit(ExitCode::INVALID_ARGUMENT);                                 \
    }

#define POLICE_UNREACHABLE()                                                    \
    POLICE_RUNTIME_ERROR(                                                       \
        "line " << __LINE__ << " of " << __FILE__                              \
                << " should not be reachable")

#define POLICE_PANIC_IF(cond, msg)                                              \
    if (cond) {                                                                \
        std::cerr << msg << std::endl;                                         \
        std::exit(ExitCode::INTERNAL_ERROR);                                   \
    }

#ifndef NDBEUG
#define POLICE_ASSERT(cond) assert(cond)
#else
#define POLICE_ASSERT(cond)
#endif

#ifndef NDEBUG
#define POLICE_REQUIRE(x) assert(x)
#else
#define POLICE_REQUIRE(x) x
#endif

#ifndef POLICE_VERBOSITY
#ifndef NDEBUG
#define POLICE_VERBOSITY 3
#else
#define POLICE_VERBOSITY 1
#endif
#endif

#if POLICE_VERBOSITY > 1
#define _POLICE_DEBUG_MSG(file, line, msg)                                      \
    std::cout << "(" << file << ":" << line << ") " << msg;
#define POLICE_DEBUG_MSG(msg) _POLICE_DEBUG_MSG(__FILE__, __LINE__, msg)
#define POLICE_DEBUG_IFMSG(cond, msg)                                           \
    if ((cond)) {                                                              \
        _POLICE_DEBUG_MSG(__FILE__, __LINE__, msg)                              \
    }
#else
#define POLICE_DEBUG_MSG(msg)
#define POLICE_DEBUG_IFMSG(cond, msg)
#endif

#if defined(_LIBCPP_VERSION)
#define POLICE_CONSTEXPR_VARIANT
#else
#define POLICE_CONSTEXPR_VARIANT constexpr
#endif
