#include "police/command_line_options.hpp"
#include "police/command_line_parser.hpp"
#include "police/execution_unit.hpp"
#include "police/global_arguments.hpp"
#include "police/utils/stopwatch.hpp"
#include "police/version.hpp"

#include <cctype>
#include <iostream>
#include <sstream>
#include <string>
#include <string_view>

namespace {
std::string stdlib_version()
{
#if defined(__GLIBCXX__)
    return std::string("libstdc++-") + std::to_string(__GLIBCXX__);
#elif defined(_LIBCPP_VERSION)
    return std::string("libc++-") + std::to_string(_LIBCPP_VERSION);
#else
    return "unrecognized lib";
#endif
}

std::string argument_string(int argc, const char** argv)
{
    std::ostringstream out;
    for (int i = 1; i < argc; ++i) {
        bool is_alph_num = true;
        std::string_view arg(argv[i]);
        for (char c : arg) {
            is_alph_num =
                is_alph_num && (std::isalnum(c) || c == '-' || c == '_');
        }
        if (i > 1) {
            out << ' ';
        }
        if (is_alph_num) {
            out << arg;
        } else {
            out << '"' << arg << '"';
        }
    }
    return out.str();
}

} // namespace

int main(int argc, const char** argv)
{
    police::ScopedStopWatch total_time("Total time");
    police::ScopedStopWatch search_time("Search time", false);

    std::cout << "police" << " revision " << GIT_HASH << " (" << COMPILER_ID
              << " " << COMPILER_VERSION << " " << stdlib_version() << ")"
              << std::endl;
    std::cout << "arguments: " << argument_string(argc, argv) << std::endl;

    police::CommandLineParser cli = police::get_command_line_options();
    police::GlobalArguments args = cli.parse(argc, argv);

    std::shared_ptr<police::ExecutionUnit> unit = args.unit;
    std::cout << "running " << unit->name() << "..." << std::endl;

    {
        police::ScopedStopWatch::ScopedTimer timer = search_time.scope();
        unit->run();
    }

    unit->report_result();
    unit->report_statistics();

    return 0;
}
