#pragma once

#include "police/lp.hpp"
#include "police/lp_factory.hpp"
#include "police/marabou_preprocessor.hpp"
#include "police/nnlp.hpp"
#include "police/nnlp_encoders.hpp"
#include "police/nnlp_factory.hpp"

#include <memory>

namespace police {

class NNLPLP final : public NNLPBase {
public:
    explicit NNLPLP(std::unique_ptr<LP> lp);

    void add_constraint(const linear_constraint_type& constraint) override;

    void add_constraint(const relu_constraint_type& constraint) override;

    void add_constraint(const max_constraint_type& constraint) override;

    void add_constraint(
        const linear_constraint_disjunction_type& constraint) override;

    [[nodiscard]]
    model_type get_model() const override;

    void dump() const override;

    bool supports_unsolvable_core() const override { return true; }

    LinearConstraintConjunction get_unsolvable_core() const override
    {
        return unsat_core_;
    }

    LP* get_underlying_lp() { return lp_.get(); }

    const LP* get_underlying_lp() const { return lp_.get(); }

private:
    void do_push_snapshot() override;
    void do_pop_snapshot() override;
    void do_clear() override;
    void do_add_variable(const VariableType& var_type) override;
    void do_set_variable_upper_bound(size_t var_ref, real_t ub) override;
    void do_set_variable_lower_bound(size_t var_ref, real_t lb) override;
    Status do_solve() override;
    Status do_solve(const vector<linear_constraint_type>& constraints) override;

    void set_variable_bounds();
    void cleanup_last_solver_call();

    LinearConstraintConjunction unsat_core_;
    std::unique_ptr<LP> lp_;
    bool dirty_state_ = false;
};

class NNLPLPFactory final : public NNLPFactory {
public:
    explicit NNLPLPFactory(LPFactory* factory, bool preprocess)
        : factory_(factory)
        , preprocess_(preprocess)
    {
    }

    [[nodiscard]]
    NNLP* make() const
    {
        if (preprocess_) {
            return new NNLPMarabouPreprocessor(std::make_shared<NNLPLP>(
                factory_->make_unique(LPOptimizationKind::MINIMIZE)));
        }
        return new NNLPLP(factory_->make_unique(LPOptimizationKind::MINIMIZE));
    }

private:
    LPFactory* factory_;
    bool preprocess_;
};

} // namespace police
